% *************************************************************************
% 
%                      iGem Modelling - 2018
%                     Delft University of Technology
% 
% *************************************************************************
% 
%               Title
% 
%                    File: Gene Doping IV and IM Model Single Dose
%                           
% 
%              
% *************************************************************************
clear all
close all
format long
tic
y1 = [0 10e11];

tset = 450;
% Integration time span
tini=0; ddt = 0.1;
t1=tini:ddt:tset;

% %Viral DNA IV Injection
% c_IV=9.4e10; %IV Injection
% c_IM= 0; %IM Injection

%Viral DNA IM Injection
c_IV=0; %IV Injection
c_IM= 1.41e11; %IM Injection

%Initial values
cbl=c_IV; cti=0; cmus= c_IM; vesicle = 0; complexintracell = 0; plasmid = 0;% cti timespan and fixed value???
plasmidbound = 0; complexbound = 0; complexboundNPC = 0; 
complexboundnucleus = 0; complexnucleus = 0; plasmidboundNPC = 0; 
plasmidboundnucleus = 0; plasmidnucleus = 0; protein = 0; proteinextra = 0;deaddna=0;

% Add inital values to vector
y0 = [cbl;cti; vesicle; complexintracell; plasmid; plasmidbound; ...
    complexbound; complexboundNPC; complexboundnucleus; complexnucleus; ...
    plasmidboundNPC; plasmidboundnucleus; plasmidnucleus; ...
    protein; proteinextra; deaddna; cmus];

% Solve system of ODEs
opt = odeset('RelTol',1e-6, 'AbsTol',1e-6);
[t_EPO, y]=ode15s(@dydt_IV_IM,tini:ddt:tset,y0,opt);

%Calculate total detectable DNA
totaldna= y(:,1) + y(:,16);
Detection_Window  = interp1(totaldna,t_EPO,1000)

%Convert EPO from from #/mL to mIU/mL
conversionfactor=3.63636752444032*10^-9;% from #/mL to mIU/mL
EPO_ex=y(:,15)*conversionfactor;
Max_EPO_level = max(EPO_ex)

%Figure of doping DNA degradation in blood and the detection window
Time_Delay_1 = -50;
figure(1)
set(gcf,'color','w'); 
plot([Time_Delay_1; t_EPO],[0; totaldna],'r', 'LineWidth',2); hold on;
plot([0 0], [0 80000],'--k' , 'LineWidth',2)
plot([Detection_Window Detection_Window], [0 80000], 'b--', 'LineWidth',2);
set(gca,'fontsize',16);
xlabel('Time (days)');
ylabel('Detectable Sequences (DNA fragments/mL)');
legend('Doping DNA in blood', 'Vector injection (day 0)',...
['DNA detection limit (day ' num2str(Detection_Window, '%4.0f') ')'])
xlim([Time_Delay_1 400])
set(gcf, 'Position', [500, 200, 800, 700])

%Time Delay before doping injection occurs
Time_Delay = 50; %Days

%Total Blood Volume
TBV = 5000; %mL

%Create maturation time span
Total_Span = 141; % Number of days from BFU-E to Erythrocytes

%Initial Cell Concentration
Cell_qin=10^8*exp(0.2*7); % cells/day, Number of mature BFU-C cells going to CFU-E cells  

%Proliferation Rates
%Units: (1/day)
B_BFUE = 0.2; %BFU-E
B_CFUE = 0.57; %CFU-E
B_Ery = 1.024; %day

% Apoptosis Rates
%Units: (1/day)
alpha_Reti = 0.089; %/day
alpha_RBCs = 0.005; %/day

%Neocytolysis Parameters
tau_E = 9.8; %mU/ml
c_E = 3.5; %mU3/(ml3day)
k_E = 3; %Dimensionless
b_E = 0.1; %1/day

% CFU-E Apoptosis Rate variables
a1 = 0.35; b1 = 0.07;
c1 = 3; k1 = 0.14;

%Bone Marrow Reticulocyte Maturation Velocity variables
a2 = 2; b2 = 0.35;
c2 = 2.3; k2 = 0.2;

%EPO Feedback loop parameters
a3 = 16500; b3 = 117000;
c3 = 9.1; k3 = 0.2;

%EPO degradation rate
c_end_deg = 25/24.;

% --- Create mesh ---
mesh = 100;
% Nodes (n), step size (h), mesh (d)
n = Total_Span*mesh; h = Total_Span/n; d = 0:h:Total_Span; % Nodes (n), step size (h), mesh (d)

%Start and End points
BFU_E_start = 0; %BFU-E start
BFU_E_end = 7-h; %BFU-E end
CFU_E_start = 7; %CFU-E start
CFU_E_end = 13-h; %CFU-E end
Erythroblasts_start = 13; %Erythroblasts start
Erythroblasts_end = 18-h; %Erythroblasts end
Reticulocyte_start = 18; %Reticulocyte start
Reticulocyte_end = 21-h; %Reticulocyte end
Erythrocytes_start1 = 21; %Erythrocytes start1
Erythrocytes_end1 = 35-h; %Erythrocytes end1
Erythrocytes_start2 = 35; %Erythrocytes start2
Erythrocytes_end2 = 42-h; %Erythrocytes end2
Erythrocytes_start3 = 42; %Erythrocytes start3
Erythrocytes_end3 = 141; %Erythrocytes end3

%Cell number at t = 0.
%Close to SS
D0 = [100000000;100200200.133400;100400801.067734;100601803.605407;100803208.550427;101005016.708417;101207228.886608;101409845.893849;101612868.540610;101816297.638979;102020134.002676;102224378.447044;102429031.789062;102634094.847344;102839568.442143;103045453.395352;103251750.530512;103458460.672812;103665584.649092;103873123.287850;104081077.419239;104289447.875076;104498235.488844;104707441.095694;104917065.532447;105127109.637602;105337574.251336;105548460.215508;105759768.373661;105971499.571029;106183654.654536;106396234.472803;106609239.876151;106822671.716599;107036530.847877;107250818.125422;107465534.406381;107680680.549622;107896257.415728;108112265.867008;108328706.767496;108545580.982955;108762889.380883;108980632.830513;109198812.202820;109417428.370521;109636482.208082;109855974.591717;110075906.399398;110296278.510851;110517091.807565;110738347.172793;110960045.491558;111182187.650653;111404774.538647;111627807.045887;111851286.064505;112075212.488415;112299587.213325;112524411.136734;112749685.157938;112975410.178032;113201587.099918;113428216.828303;113655300.269706;113882838.332462;114110831.926724;114339281.964465;114568189.359486;114797555.027418;115027379.885723;115257664.853701;115488410.852491;115719618.805080;115951289.636297;116183424.272828;116416023.643211;116649088.677844;116882620.308987;117116619.470767;117351087.099181;117586024.132100;117821431.509272;118057310.172328;118293661.064781;118530485.132037;118767783.321391;119005556.582036;119243805.865067;119482532.123480;119721736.312181;119961419.387987;120201582.309630;120442226.037763;120683351.534961;120924959.765725;121167051.696490;121409628.295623;121652690.533432;121896239.382164;122140275.816017;122384800.811136;122629815.345621;122875320.399531;123121316.954887;123367805.995674;123614788.507850;123862265.479345;124110237.900067;124358706.761906;124607673.058738;124857137.786428;125107101.942836;125357566.527819;125608532.543234;125860000.992948;126111972.882833;126364449.220778;126617431.016688;126870919.282491;127124915.032140;127379419.281619;127634433.048945;127889957.354174;128145993.219402;128402541.668774;128659603.728484;128917180.426780;129175272.793970;129433881.862424;129693008.666577;129952654.242938;130212819.630089;130473505.868693;130734714.001494;130996445.073325;131258700.131111;131521480.223872;131784786.402730;132048619.720910;132312981.233744;132577871.998679;132843293.075279;133109245.525229;133375730.412338;133642748.802547;133910301.763929;134178390.366697;134447015.683205;134716178.787955;134985880.757600;135256122.670948;135526905.608967;135798230.654789;136070098.893715;136342511.413218;136615469.302948;136888973.654738;137163025.562604;137437626.122756;137712776.433596;137988477.595725;138264730.711948;138541536.887279;138818897.228941;139096812.846378;139375284.851252;139654314.357451;139933902.481093;140214050.340532;140494759.056359;140776029.751410;141057863.550768;141340261.581768;141623224.974002;141906754.859326;142190852.371858;142475518.647989;142760754.826385;143046562.047990;143332941.456034;143619894.196035;143907421.415805;144195524.265452;144484203.897388;144773461.466332;145063298.129316;145353715.045685;145644713.377109;145936294.287580;146228458.943422;146521208.513296;146814544.168199;147108467.081474;147402978.428814;147698079.388264;147993771.140229;148290054.867475;148586931.755139;148884402.990728;149182469.764127;149481133.267604;149780394.695814;150080255.245802;150380716.117011;150681778.511285;150983443.632875;151285712.688439;151588586.887057;151892067.440224;152196155.561863;152500852.468328;152806159.378406;153112077.513325;153418608.096758;153725752.354828;154033511.516113;154341886.811649;154650879.474938;154960490.741951;155270721.851134;155581574.043411;155893048.562192;156205146.653374;156517869.565352;156831218.549017;157145194.857765;157459799.747502;157775034.476648;158090900.306142;158407398.499448;158724530.322560;159042297.044004;159360699.934848;159679740.268705;159999419.321736;160319738.372657;160640698.702746;160962301.595844;161284548.338362;161607440.219289;161930978.530193;162255164.565226;162579999.621134;162905484.997257;163231621.995538;163558411.920524;163885856.079376;164213955.781871;164542712.340407;164872127.070013;165202201.288346;165532936.315705;165864333.475031;166196394.091911;166529119.494589;166862511.013967;167196569.983611;167531297.739759;167866695.621321;168202764.969889;168539507.129741;168876923.447846;169215015.273871;169553783.960182;169893230.861855;170233357.336678;170574164.745158;170915654.450523;171257827.818735;171600686.218486;171944231.021211;172288463.601089;172633385.335051;172978997.602785;173325301.786740;173672299.272133;174019991.446954;174368379.701974;174717465.430745;175067250.029610;175417734.897709;175768921.436982;176120811.052174;176473405.150846;176826705.143374;177180712.442957;177535428.465627;177890854.630248;178246992.358524;178603843.075007;178961408.207101;179319689.185066;179678687.442027;180038404.413978;180398841.539786;180760000.261200;181121882.022857;181484488.272284;181847820.459905;182211880.039051;182576668.465960;182942187.199786;183308437.702605;183675421.439419;184043139.878164;184411594.489713;184780786.747887;185150718.129454;185521390.114140;185892804.184634;186264961.826593;186637864.528647;187011513.782409;187385911.082474;187761057.926434;188136955.814876;188513606.251392;188891010.742585;189269170.798072;189648087.930495;190027763.655523;190408199.491858;190789396.961245;191171357.588475;191554082.901390;191937574.430891;192321833.710947;192706862.278594;193092661.673946;193479233.440203;193866579.123652;194254700.273675;194643598.442759;195033275.186496;195423732.063594;195814970.635881;196206992.468311;196599799.128973;196993392.189093;197387773.223045;197782943.808353;198178905.525699;198575659.958933;198973208.695070;199371553.324308;199770695.440025;200170636.638790;200571378.520369;200972922.687729;201375270.747048;201778424.307718;202182384.982354;202587154.386801;202992734.140134;203399125.864675;203806331.185991;204214351.732903;204623189.137494;205032845.035115;205443321.064389;205854618.867221;206266740.088803;206679686.377621;207093459.385460;207508060.767412;207923492.181884;208339755.290603;208756851.758620;209174783.254322;209593551.449437;210013158.019036;210433604.641548;210854892.998759;211277024.775823;211700001.661268;212123825.347001;212548497.528319;212974019.903911;213400394.175866;213827622.049682;214255705.234271;214684645.441968;215114444.388532;215545103.793160;215976625.378492;216409010.870612;216842261.999065;217276380.496855;217711368.100456;218147226.549820;218583957.588381;219021562.963064;219460044.424291;219899403.725988;220339642.625594;220780762.884063;221222766.265879;221665654.539054;222109429.475143;222554092.849247;222999646.440018;223446092.029673;223893431.403993;224341666.352338;224790798.667647;225240830.146451;225691762.588875;226143597.798651;226596337.583120;227049983.753241;227504538.123599;227960002.512414;228416378.741543;228873668.636491;229331874.026418;229790996.744148;230251038.626171;230712001.512656;231173887.247454;231636697.678109;232100434.655864;232565100.035667;233030695.676181;233497223.439787;233964685.192599;234433082.804464;234902418.148972;235372693.103466;235843909.549047;236316069.370580;236789174.456705;237263226.699844;237738227.996207;238214180.245798;238691085.352428;239168945.223717;239647761.771107;240127536.909863;240608272.559086;241089970.641721;241572633.084560;242056261.818253;242540858.777316;243026425.900138;243512965.128988;244000478.410022;244488967.693296;244978434.932766;245468882.086303;245960311.115695;246452723.986660;246946122.668849;247440509.135858;247935885.365234;248432253.338482;248929615.041074;249427972.462458;249927327.596065;250427682.439316;250929038.993630;251431399.264434;251934765.261171;252439138.997305;252944522.490332;253450917.761786;253958326.837248;254466751.746357;254976194.522812;255486657.204385;255998141.832927;256510650.454378;257024185.118773;257538747.880251;258054340.797064;258570965.931585;259088625.350313;259607321.123889;260127055.327095;260647830.038870;261169647.342312;261692509.324692;262216418.077457;262741375.696245;263267384.280886;263794445.935415;264322562.768080;264851736.891348;265381970.421917;265913265.480721;266445624.192942;266979048.688015;267513541.099638;268049103.565783;268585738.228699;269123447.234926;269662232.735301;270202096.884967;270743041.843380;271285069.774322;271828182.845905;272372383.230581;272917673.105153;273464054.650782;274011530.052994;274560101.501692;275109771.191161;275660541.320083;276212414.091537;276765391.713015;277319476.396430;277874670.358120;278430975.818862;278988395.003878;279546930.142846;280106583.469908;280667357.223677;281229253.647249;281792274.988211;282356423.498648;282921701.435156;283488111.058847;284055654.635360;284624334.434870;285194152.732098;285765111.806316;286337213.941363;286910461.425648;287484856.552161;288060401.618483;288637098.926796;289214950.783889;289793959.501171;290374127.394678;290955456.785081;291537949.997700;292121609.362507;292706437.214141;293292435.891914;293879607.739822;294467955.106552;295057480.345496;295648185.814755;296240073.877150;296833146.900237;297427407.256307;298022857.322402;298619499.480324;299217336.116642;299816369.622704;300416602.394643;301018036.833393;301620675.344692;302224520.339094;302829574.231980;303435839.443568;304043318.398917;304652013.527946;305261927.265434;305873062.051039;306485420.329300;307099004.549651;307713817.166430;308329860.638887;308947137.431198;309565650.012471;310185400.856757;310806392.443060;311428627.255346;312052107.782557;312676836.518616;313302815.962437;313930048.617940;314558536.994056;315188283.604739;315819290.968977;316451561.610800;317085098.059291;317719902.848597;318355978.517939;318993327.611619;319631952.679034;320271856.274687;320913040.958191;321555509.294288;322199263.852850;322844307.208897;323490641.942604;324138270.639310;324787195.889530;325437420.288967;326088946.438519;326741776.944292;327395914.417608;328051361.475018;328708120.738312;329366194.834527;330025586.395960;330686298.060179;331348332.470031;332011692.273655;332676380.124490;333342398.681290;334009750.608129;334678438.574415;335348465.254902;336019833.329698;336692545.484274;337366604.409482;338042012.801557;338718773.362134;339396888.798255;340076361.822385;340757195.152415;341439391.511681;342122953.628967;342807884.238525;343494186.080076;344181861.898830;344870914.445490;345561346.476268;346253160.752892;346946360.042621;347640947.118253;348336924.758137;349034295.746184;349733062.871880;350433228.930293;351134796.722090;351837769.053541;352542148.736538;353247938.588600;353955141.432887;354663760.098212;355373797.419050;356085256.235552;356798139.393554;357512449.744588;358228190.145899;358945363.460447;359663972.556928;360384020.309779;361105509.599192;361828443.311124;362552824.337312;363278655.575281;364005939.928356;364734680.305677;365464879.622205;366196540.798739;366929666.761924;367664260.444266;368400324.784140;369137862.725804;369876877.219411;370617371.221020;371359347.692608;372102809.602082;372847759.923291;373594201.636036;374342137.726086;375091571.185186;375842505.011071;376594942.207478;377348885.784155;378104338.756878;378861304.147461;379619784.983765;380379784.299715;381141305.135310;381904350.536634;382668923.555868;383435027.251308;384202664.687367;384971838.934598;385742553.069697;386514810.175524;387288613.341106;388063965.661658;388840870.238590;389619330.179522;390399348.598293;391180928.614980;391964073.355902;392748785.953641;393535069.547047;394322927.281256;395112362.307700;395903377.784120;396695976.874579;397490162.749475;398285938.585551;399083307.565913;399882272.880037;400682837.723786;401485005.299420;402288778.815611;403094161.487453;403901156.536479;404709767.190669;405519996.684468;406599909.571259;407682698.363304;408768370.709683;409856934.279662;410948396.762747;412042765.868734;413140049.327769;414240254.890416;415343390.327705;416449463.431185;417558482.012978;418670453.905829;419785386.963150;420903289.059058;422024168.088428;423148031.994059;424274888.740192;425404746.312567;426537612.718500;427673495.986952;428812404.168599;429954345.335885;431099327.583094;432247359.026408;433398447.776182;434552601.964186;435709829.743651;436870139.289332;438033538.797543;439200036.486207;440369640.594925;441542359.385064;442718201.139807;443897174.164211;445079286.785262;446264547.351930;447452964.235214;448644545.856936;449839300.661657;451037237.116742;452238363.712426;453442688.961883;454650221.401295;455860969.589940;457074942.110255;458292147.538530;459512594.473725;460736291.537512;461963247.374324;463193470.651412;464426970.058899;465663754.309845;466903832.140310;468147212.309433;469393903.599482;470643914.815923;471897254.787472;473153932.366144;474413956.457699;475677335.991939;476944079.922755;478214197.228215;479487696.910642;480764587.996686;482044879.537407;483328580.608358;484615700.278593;485906247.641142;487200231.813070;488497661.935541;489798547.173876;491102896.717604;492410719.780510;493722025.600698;495036823.440648;496355122.587284;497676932.352028;499002262.070870;500331121.136490;501663518.966955;502999465.005801;504338968.722110;505682039.610585;507028687.191620;508378921.011379;509732750.609181;511090185.549538;512451235.422222;513815909.842328;515184218.450335;516556170.912176;517931776.919299;519311046.188744;520693988.463230;522080613.511213;523470931.126954;524864951.130588;526262683.368171;527664137.745552;529069324.195343;530478252.676988;531890933.176848;533307375.708274;534727590.311700;536151587.054737;537579376.032250;539010967.331889;540446371.067981;541885597.381585;543328656.440546;544775558.439564;546226313.600259;547680932.171245;549139424.428190;550601800.673893;552068071.238350;553538246.478837;555012336.779980;556490352.553810;557972304.239831;559458202.340966;560948057.388541;562441879.942376;563939680.590846;565441469.950947;566947258.668383;568457057.417649;569970876.902108;571488727.817443;573010620.887598;574536566.864847;576066576.529867;577600660.691809;579138830.188373;580681095.885896;582227468.679437;583777959.492860;585332579.278896;586891339.019223;588454249.724545;590021322.434657;591592568.218524;593167998.212360;594747623.582472;596331455.525352;597919505.267759;599511784.066805;601108303.210048;602709074.015575;604314107.793356;605923415.883237;607537009.655032;609154900.508582;610777099.873844;612403619.210964;614034470.010369;615669663.792861;617309212.109699;618953126.542673;620601418.704184;622254100.237322;623911182.855937;625572678.305573;627238598.363552;628908954.839036;630583759.573132;632263024.438974;633946761.341819;635634982.178391;637327698.876860;639024923.396911;640726667.729838;642432943.898616;644143763.957981;645859139.994519;647579084.126754;649303608.505261;651032725.312716;652766446.763970;654504785.106124;656247752.618638;657995361.613423;659747624.434908;661504553.502510;663266161.269213;665032460.221671;666803462.880294;668579181.799355;670359629.567083;672144818.805764;673934762.171837;675729472.312671;677528961.909066;679333243.675336;681142330.359397;682956234.742868;684774969.641179;686598547.903658;688426982.413616;690260286.088436;692098471.879663;693941552.773084;695789541.788825;697642451.981414;699500296.439854;701363088.287705;703230840.728257;705103567.000493;706981280.379193;708863994.175043;710751721.734728;712644476.441046;714542271.713004;716445120.960002;718353037.626880;720266035.194016;722184127.177414;724107327.128799;726035648.635699;727969105.321568;729907710.845889;731851478.904271;733800423.228541;735754557.586837;737713895.783678;739678451.660045;741648239.141027;743623272.189387;745603564.805655;747589131.028228;749579984.933489;751576140.635909;753577612.239853;755584413.886965;757596559.756271;759614064.064270;761636941.065032;763665205.050295;765698870.349556;767737951.330178;769782462.397468;771832417.994793;773887832.603669;775948720.793594;778015097.173472;780086976.391744;782164373.136519;784247302.135700;786335778.106678;788429815.805794;790529430.028433;792634635.609134;794745447.421663;796861880.379089;798983949.433895;801111669.578066;803245055.843194;805384123.300588;807528887.113128;809679362.484707;811835564.660333;813997508.926235;816165210.610016;818338685.028300;820517947.538207;822703013.537446;824893898.464433;827090617.798383;829293187.059418;831501621.808678;833715937.648413;835936150.222124;838162275.214678;840394328.352430;842632325.457332;844876282.394157;847126215.070626;849382139.437542;851644071.488938;853912027.207410;856186022.617811;858466073.787369;860752196.825775;863044407.885309;865342723.160939;867647158.890433;869957731.354471;872274456.876744;874597351.824082;876926432.662763;879261715.903610;881603218.102125;883950955.858604;886304945.818280;888665204.614455;891031748.924350;893404595.469259;895783761.014683;898169262.370435;900561116.390785;902959339.974595;905363950.065436;907774963.651708;910192397.766740;912616269.488926;915046595.941825;917483394.353071;919926681.996913;922376476.194328;924832794.313170;927295653.768306;929765072.021742;932241066.522999;934723654.767752;937212854.297932;939708682.701858;942211157.614350;944720296.716853;947236117.737554;949758638.451496;952287876.680703;954823850.294304;957366577.208639;959916075.387401;962472362.841737;965035457.692269;967605378.108694;970182142.309924;972765768.564239;975356275.189413;977953680.490172;980558002.819687;983169260.579642;985787472.220386;988412656.241048;991044831.189658;993684015.663273;996330228.308110;998983487.819634;1001643813.00688;1004311222.72970;1006985735.89900;1009667371.47686;1012356148.41177;1015052085.70235;1017755202.39746;1020465517.59629;1023183050.44857;1025907820.15461;1028639845.96548;1031379147.18317;1034125743.16073;1036879653.30239;1039640897.13040;1042409494.21985;1045185464.19889;1047968826.74882;1050759601.60422;1053557808.48559;1056363467.16556;1059176597.46906;1061997219.27347;1064825352.50870;1067661017.15737;1070504233.25492;1073355020.88978;1076213400.20345;1079079391.39068;1081953014.69960;1084834290.43185;1087723239.01242;1090619880.92159;1093524236.69507;1096436326.92413;1099356172.25581;1102283793.32238;1105219210.81068;1108162445.46226;1111113518.07351;1114072449.49585;1117039260.63587;1120013972.45551;1122996605.97215;1125987182.25874;1128985722.44390;1131992247.78469;1135006779.59567;1138029339.24911;1141059948.17513;1144098627.78856;1147145399.56082;1150200285.02015;1153263305.75170;1156334483.39769;1159413839.65757;1162501396.28814;1165597175.10373;1168701197.97632;1171813486.91082;1174934063.97168;1178062951.28309;1181200171.02914;1184345745.37804;1187499696.55666;1190662046.85062;1193832818.60450;1197012034.22193;1200199716.16581;1203395886.95838;1206600569.18147;1209813785.47658;1213035558.54516;1216265911.22666;1219504866.42243;1222752447.09586;1226008676.27256;1229273577.04049;1232547172.47111;1235829485.69692;1239120539.91162;1242420358.37024;1245728964.38932;1249046381.34707;1252372632.68347;1255707741.90044;1259051732.56191;1262404628.29412;1265766452.78573;1269137229.78803;1272516983.11507;1275905736.72560;1279303514.64325;1282710340.95663;1286126239.81956;1289551235.45128;1292985352.13658;1296428614.14294;1299881045.80203;1303342671.50991;1306813515.72725;1310293602.97949;1313782957.85699;1317281605.01523;1320789569.17491;1324306875.12212;1327833547.70847;1331369611.85125;1334915092.53367;1338470014.80494;1342034403.78049;1345608284.64212;1349191682.72468;1352784623.43155;1356387132.23489;1359999234.67579;1363620956.36453;1367252322.98069;1370893360.18556;1374544093.70830;1378204549.34615;1381874752.96454;1385554730.49737;1389244507.94705;1392944111.38467;1396653566.95021;1400372900.85270;1404102139.37046;1407841308.85128;1411590435.71258;1415349546.44161;1419118667.68658;1422897826.16782;1426687048.67796;1430486362.08219;1434295793.31842;1438115369.39752;1441945117.31106;1445785064.12205;1449635236.96506;1453495663.04650;1457366369.64481;1461247384.11066;1465138733.86712;1469040446.40988;1472952549.30736;1476875070.20090;1480808036.80489;1484751476.90701;1488705418.36839;1492669889.12380;1496644917.18185;1500630530.72134;1504626757.99737;1508633627.34146;1512651167.16188;1516679405.94378;1520718372.24948;1524768094.62091;1528828601.67550;1532899922.10639;1536982084.68266;1541075118.24942;1545179051.72806;1549293914.11638;1553419734.48877;1557556541.99643;1561704365.86763;1565863235.40795;1570033180.00039;1574214229.10562;1578406412.26218;1582609759.18811;1586824299.68187;1591050063.62258;1595287080.97020;1599535381.76586;1603794996.13202;1608065954.16964;1612348286.05931;1616642022.06147;1620947192.51658;1625263827.84536;1629591958.54899;1633931615.20934;1638282828.48924;1642645629.13269;1647020047.96505;1651406115.89325;1655803863.90590;1660213323.07353;1664634524.54879;1669067499.67364;1673512279.87489;1677968896.66437;1682437381.63925;1686917766.48226;1691410082.96186;1695914362.82378;1700430637.89765;1704958940.09724;1709499301.42072;1714051753.95094;1718616329.85557;1723193061.38735;1727781980.88432;1732383120.77000;1736996513.55367;1741622191.83053;1746260188.28194;1750910535.67566;1755573266.86603;1760248414.90708;1764936012.94225;1769636094.20470;1774348692.01756;1779073839.79419;1783811571.03843;1788561919.23022;1793324917.93807;1798100600.81930;1802889001.62025;1807690154.17654;1812504092.41325;1817330850.34520;1822170462.07714;1827022961.80399;1831888383.81113;1836766762.47451;1841658132.26099;1846562527.72854;1851479983.52644;1856410534.39555;1861354215.28785;1866311061.24989;1871281107.42301;1876264389.04357;1881260941.44324;1886270800.04927;1891294000.26368;1896330577.58209;1901380567.59401;1906444005.98310;1911520928.52736;1916611371.09942;1921715369.66676;1926832960.29200;1931964179.13321;1937109062.44416;1942267646.57449;1947439967.96992;1952626063.29767;1957825969.32416;1963039722.91535;1968267361.03703;1973508920.62852;1978764438.72690;1984033952.46718;1989317499.08253;1994615115.90464;1999926840.36383;2020511303.45790;2041307634.01150;2062318012.82827;2083544643.16379;2104989750.82191;2126655584.51629;2148544416.10607;2170658540.83405;2193000277.56745;2215571969.04112;2238375982.10318;2261414707.96329;2284690562.58983;2308205986.82453;2331963446.63857;2355965433.24060;2380214463.32871;2404713079.49905;2429463850.51221;2454469371.56226;2479732264.54864;2505255178.35089;2531040789.10612;2557091800.48944;2583410944.16279;2610000979.90423;2636864696.06633;2664004909.70832;2691424466.89182;2719126242.97965;2747113142.76145;2775388100.92529;2803954082.36556;2832814082.49376;2861971127.55268;2891428274.93372;2921188613.49763;2951255263.70933;2981631378.33563;3012320142.59432;3043324774.48819;3074648524.94545;3106294678.34735;3138266552.87206;3170567500.84234;3203200909.28211;3236170200.07670;3269478830.54131;3303130293.58493;3337128117.86307;3371475868.35092;3406177146.71724;3441235591.92247;3476654880.16831;3512438725.49479;3548590880.39728;3585115135.77445;3622015321.54420;3659295306.81089;3696959000.73049;3735010352.69559;3773453352.50758;3812292031.02435;3851530460.58257;3891172755.42426;3931223072.12767;3971685610.29739;4012564613.02004;4053864367.06533;4095589203.33593;4137743497.05671;4180331668.48586;4223358183.37858;4266827553.45534;4310744336.87487;4355113138.71217;4399938611.44138;4445225455.70862;4490978420.26692;4537202302.45669;4583901948.98440;4631082256.72720;4678748172.66454;4726904694.68144;4775556872.09190;4824709806.78649;4874368652.86805;4924538617.48904;4975224961.71643;5026433000.45887;5078168103.33003;5130435695.54031;5183241257.82169;5236590327.31788;5290488498.16503;5344941422.42084;5399954810.33253;5455534430.58653;5511686110.89256;5568415738.92966;5625729262.96311;5683632692.46754;5742132098.75668;5801233615.99091;5860943441.46732;5921267836.26950;5982213125.92401;6043785701.06349;6105992018.09724;6168838599.88832;6232332036.43758;6296478985.57446;6361286173.65526;6426760396.68043;6492908520.61666;6559737482.11757;6627254288.82681;6695466020.51633;6764379829.82908;6834002943.02895;6904342660.31595;6975406357.01117;7047201484.32934;7119735570.61614;7193016221.70428;7267051121.71118;7341848033.37406;7417414799.31032;7493759343.31949;7570889670.75775;7648813869.37752;7727540110.17547;7807076648.74970;7887431825.69090;7968614067.45772;8050631886.74475;8133493883.86544;8217208747.65387;8301785256.37582;8387232278.64938;8473558774.37487;8560773795.67432;8648886487.28664;8737906088.61248;8827841934.15115;8918703454.47947;9010500177.24073;9103241727.56036;9196937829.60850;9291598308.21404;9387233089.32898;9483852201.06941;9581465774.76738;9680084046.03322;9779717355.82883;9880376151.55194;9982070988.77071;10084812531.7243;10188611554.4421;10293478941.8745;10399425690.3694;10506462908.7849;10614601820.2919;10723853763.5507;10834230193.8990;10945742684.5523;11058402927.8158;11172222736.3102;11287214044.9316;11403388912.1465;11520759520.5640;11639338178.2138;11759137319.8387;11880169507.4381;12002447432.3085;12125983916.3738;12250791913.5295;12376884511.0017;12504274930.7196;12632976530.7018;12763002806.4569;12894367392.3982;13027084064.1087;13161166738.1506;13296629474.3260;13433486477.1513;13571752096.4769;13711440829.8166;13852567323.8666;13995146376.0390;14139192936.0125;14284722108.2149;14431749152.5371;14580289485.9330;14730358684.9801;14881972486.6173;15035146789.7964;15189897657.1502;15346241315.6953;15504194159.4676;15663772751.2413;15824993824.2657;15987874284.0199;16152431209.9858;16318681857.4399;16486643659.2620;16656334226.6959;16827771352.2074;17000973012.4387;17175957369.0589;17352742770.6683;17531347754.7227;17711791049.4769;17894091575.9494;18078268449.9056;18264340983.8625;18452328689.1139;18642251277.7765;18834128664.8578;19027980970.3438;19223828521.3077;19421691854.0408;19621591716.2062;19823549069.0138;20027585089.4189;20233721173.6404;20441978936.8628;20652380216.7469;20864947075.7206;21079701803.2923;21296666918.3893;21515865170.3398;21737319543.9600;21961053260.6402;22187089780.7798;22415452806.2477;22646166282.8674;22879254402.9278;23114741607.7203;23352652590.1016;23593012297.0830;23835845932.4459;24081178959.3856;24329037101.6207;24579446349.1451;24832432959.4563;25088023460.3076;25346244652.4908;25607123614.2868;25870687702.7491;26136964554.8996;26405982092.2173;26677768523.5673;26952352348.1590;27229762358.5346;27510027643.5888;27793177591.6181;28079241893.4037;28368250543.5052;28660233845.1313;28955222413.3139;29253247178.1169;29554339387.8740;29858530616.2894;30165852762.1149;30476338052.4992;30790019046.3696;31106928637.8505;31427100059.7156;31750566884.8415;32077363031.6604;32407522767.7161;32741080713.2566;33078071844.8634;33418531496.9781;33762495367.6391;34109999522.2212;34461080397.2145;34815774804.0418;35174119935.1717;35536153365.8758;35901913058.1685;36271437367.1106;36644765042.6281;37021935233.5786;37402987491.8597;37787961776.5613;38176898458.1579;38569838322.7473;38966822571.3354;39367892828.8973;39773091148.7384;40182460016.8999;40596042356.6098;41013881535.4108;41436021367.2109;41862506116.8751;42293380504.8655;42728689711.9300;43168479383.8382;43612795636.1673;44061685059.1397;44515194722.5065;44973372180.4847;45436265476.7416;45903923152.3750;46376394245.2412;46853728297.9225;47335975362.9214;47823186007.9104;48315411321.0330;48812702916.2612;49315112941.9691;49822694080.2111;50335499555.2775;50853583139.2759;51376999157.7695;51905802495.4746;52440048602.0134;52979793497.7315;53525093779.5689;54076006626.9974;54632589808.0146;55194901685.2019;55763001221.8466;56336947984.5114;56916802152.7038;57502624525.1786;58094476526.3060;58692420212.5071;59296518286.3545;59906834095.9358;60523431641.5009;61146375582.1780;61775731242.7577;62411564620.5508;63053942392.3148;63702931921.2546;64358601259.9692;65021019157.3405;65690255069.6921;66366379168.0675;67049462345.5771;67739576229.1695;68436793183.0185;69141186316.1120;69852829489.9132;70571797326.1093;71298165214.4344;72032009320.5728;72773406594.1458;73522434776.7779;74279172410.2466;75043698839.9093;75816094232.4385;76596439579.6180;77384816706.8339;78181308281.6566;78985997827.5749;79798969727.9497;80620309234.8740;81450102478.1184;82288436468.8977;83135399114.0017;83991079225.0120;84855566527.6106;85728951670.9887;86611326237.3538;87502782751.5307;88403414690.6701;89313316488.3229;90232583549.7725;91161312262.0271;92099600003.9217;93047545156.3149;94005247118.4274;94972806312.5387;95950324194.5147;96937903264.4446;97935647083.6610;98943660279.6437;99962048557.0026;100990918708.573;102030378626.623;103080537314.184;104141504889.806;105213392605.454;106296312851.356;107390379174.251;108495706296.238;109612410120.199;110740607741.954;111880417462.535;113031958800.595;114195352504.941;115370720567.198;116558186234.602;117757874022.913;118969909721.852;120194420415.515;121431534495.682;122681381675.271;123944093017.806;125219800936.092;126508639206.109;127810742981.055;129126248805.529;130455294629.861;131798019824.593;133154565195.101;134525072987.754;135909686913.014;137308552151.697;138721815378.553;140149624777.618;141592130057.754;143049482468.328;144521834824.328;146009341513.580;147512158512.943;149030443404.656;150564355392.867;152114055320.321;153679705685.235;155261470658.335;156859516100.073;158474009578.014;160105120394.658;161753019595.465;163417879986.801;165099876143.493;166799184437.175;168515983054.789;170250452017.270;172002773198.427;173773130344.016;175561709091.009;177368696987.059;179194283510.158;181038660088.511;182902020120.604;184784558995.483;186686474113.230;188607964905.672;190549232857.289;192510481526.351;194491916566.257;196493745747.099;198516178977.445;200559428326.355;202623708045.615;204709234579.075;206816226597.798;208944905022.957;211095493062.504;213268216221.732;215463302326.917;217680981563.161;219921486485.296;222185052042.288;224471915601.893;226782316975.572;229116498443.661;231474704765.960;233857183221.495;236264183634.432;238695958400.279;241152762512.353;243634853588.516;246142491898.202;248675940373.751;251235464653.103;253821333107.627;256433816870.232;259073189863.776;261739728829.752;264433713374.245;267155425981.130;269905152041.702;272683179884.589;275489800823.616;278325309171.623;281190002271.350;284084180526.656;287008147434.051;289962209614.549;292946676845.840;295961862094.797;299008081550.333;302085654637.225;305194904067.994;308336155856.972;311509739373.218;314715987375.027;317955236044.782;321227825024.192;324534097449.869;327874400010.300;331249082962.234;334658500167.402;334360786736.793;334063338111.902;333766154055.752;333469234331.588;333172578702.864;332876186933.249;332580058786.604;332284194048.270;331988592503.939;331693253939.647;331398178141.779;331103364875.847;330808813907.569;330514525002.858;330220497927.836;329926732448.824;329633228332.341;329339985345.128;329047003254.125;328754281826.472;328461820829.509;328169620030.779;327877679198.022;327585998099.171;327294576502.339;327003414175.823;326712510888.090;326421866407.787;326131480503.759;325841352945.052;325551483500.903;325261871919.906;324972517950.696;324683421341.966;324394581842.449;324105999200.927;323817673166.233;323529603528.718;323241790079.089;322954232608.387;322666930907.988;322379884769.610;322093093985.300;321806558347.441;321520277648.764;321234251682.328;320948480241.543;320662963120.156;320377700091.737;320092690930.053;319807935409.069;319523433282.446;319239184303.885;318955188227.136;318671444805.996;318387953794.316;318104714945.996;317821728035.355;317538992836.903;317256509125.337;316974276675.543;316692295262.595;316410564661.762;316129084648.519;315847854998.548;315566875487.717;315286145892.094;315005665987.918;314725435551.596;314445454379.850;314165722269.736;313886239018.629;313607004404.139;313328018204.061;313049280196.389;312770790159.323;312492547871.273;312214553110.836;311936805676.806;311659305368.294;311382051964.803;311105045246.015;310828284991.786;310551770982.155;310275502997.349;309999480817.778;309723704224.024;309448172996.837;309172886917.155;308897845766.098;308623049305.197;308348497296.030;308074189500.222;307800125679.443;307526305615.124;307252729088.871;306979395882.465;306706305777.849;306433458557.141;306160854022.242;305888491975.369;305616372199.472;305344494477.686;305072858593.350;304801464349.531;304530311549.639;304259399997.417;303988729496.942;303718299852.618;303448110849.723;303178162273.731;302908453910.293;302638985525.849;302369756886.878;302100767759.912;301832017911.526;301563507127.674;301295235194.487;301027201898.280;300759407025.544;300491850362.952;300224531697.356;299957450815.776;299690607505.407;299424001553.615;299157632747.951;298891500876.118;298625605725.983;298359947085.597;298094524743.182;297829338506.232;297564388182.545;297299673580.234;297035194507.727;296770950773.759;296506942168.396;296243168481.888;295979629504.681;295716325008.447;295453254764.907;295190418545.831;294927816141.934;294665447344.120;294403311943.486;294141409731.320;293879740499.094;293618304038.458;293357100122.435;293096128524.091;292835389035.308;292574881448.131;292314605573.502;292054561203.943;291794748150.828;291535166225.821;291275815222.215;291016694952.135;290757805228.001;290499145862.548;290240716650.212;289982517367.036;289724547807.682;289466807766.987;289209297039.960;288952015421.793;288694962707.851;288438138693.672;288181543156.501;287925175873.631;287669036622.399;287413125198.623;287157441398.276;286901985017.488;286646755852.552;286391753699.932;286136978374.608;285882429691.868;285628107448.997;285374011443.452;285120141491.112;284866497408.143;284613078992.752;284359886061.537;284106918413.202;283854175846.626;283601658142.685;283349365100.480;283097296519.274;282845452180.382;282593831883.269;282342435427.572;282091262595.023;281840313185.472;281589586998.936;281339083835.594;281088803477.784;280838745707.879;280588910344.246;280339297205.537;280089906110.699;279840736878.967;279591789329.872;279343063283.239;279094558541.298;278846274906.445;278598212163.396;278350370114.758;278102748563.314;277855347312.016;277608166163.984;277361204922.505;277114463391.044;276867941373.237;276621638655.161;276375555022.937;276129690262.723;275884044178.403;275638616574.019;275393407271.411;275148416092.705;274903642860.312;274889897957.208;274876153775.802;274862410315.388;274848667575.261;274834925537.101;274821184200.014;274807443563.096;274793703625.435;274779964386.106;274766225844.179;274752487998.712;274738750831.135;274725014322.677;274711278454.351;274697543224.568;274683808631.706;274670074691.723;274656341420.739;274642608835.039;274628876951.081;274615145785.490;274601415355.079;274587685659.243;274573956679.811;274560228398.429;274546500796.550;274532773873.031;274519047626.717;274505322056.443;274491597161.056;274477872939.393;274464149390.282;274450426512.535;274436704304.952;274422982783.895;274409261948.301;274395541797.081;274381822329.121;274368103543.270;274354385455.928;274340668066.088;274326951372.737;274313235374.847;274299520071.379;274285805461.288;274272091543.514;274258378316.988;274244665780.635;274230953915.821;274217242703.740;274203532142.954;274189822232.011;274176112987.010;274162404424.228;274148696560.099;274134989411.239;274121282976.875;274107577256.229;274093872248.518;274080167952.949;274066464368.717;274052761477.451;274039059260.591;274025357699.373;274011656774.830;273997956485.352;273984256846.839;273970557875.356;273956859587.149;273943161998.643;273929465126.439;273915768987.318;273902073580.699;273888378888.442;273874684892.224;273860991573.520;273847298913.625;273833606911.182;273819915564.820;273806224873.155;273792534834.788;273778845448.309;273765156729.816;273751468695.575;273737781362.024;273724094745.781;273710408863.637;273696723715.019;273683039281.814;273669355545.721;273655672488.256;273641990090.753;273628308351.882;273614627270.304;273600946844.665;273587267073.599;273573587955.712;273559909489.588;273546231691.289;273532554577.041;273518878163.235;273505202466.435;273491527503.368;273477853290.923;273464179811.116;273450507045.773;273436834976.550;273423163584.936;273409492852.247;273395822759.612;273382153305.487;273368484488.311;273354816306.504;273341148775.972;273327481912.784;273313815733.183;273300150253.572;273286485490.529;273272821443.310;273259158111.158;273245495493.312;273231833589.005;273218172397.457;273204511900.369;273190852079.242;273177192915.363;273163534407.300;273149876553.600;273136219352.802;273122562820.948;273108906974.251;273095251811.613;273081597331.918;273067943534.046;273054290416.861;273040637979.213;273026986219.949;273013335137.896;272999684731.871;272986035000.682;272972385960.594;272958737628.057;272945090002.214;272931443082.211;272917796867.209;272904151356.352;272890506531.294;272876862373.489;272863218881.682;272849576054.611;272835933890.993;272822292389.538;272808651548.941;272795011367.883;272781371845.031;272767732996.503;272754094838.573;272740457387.676;272726820660.395;272713184673.485;272699549426.414;272685914901.171;272672281079.562;272658647943.200;272645015473.513;272631383651.733;272617752458.891;272604121875.839;272590491918.151;272576862601.555;272563233941.927;272549605955.294;272535978657.835;272522352048.447;272508726126.008;272495100889.398;272481476337.497;272467852469.192;272454229283.371;272440606778.920;272426984972.151;272413363879.550;272399743517.769;272386123903.639;272372505019.275;272358886846.615;272345269367.423;272331652563.286;272318036415.592;272304420905.545;272290806031.577;272277191792.106;272263578202.984;272249965262.791;272236352970.103;272222741323.472;272209130338.866;272195520032.410;272181910402.960;272168301449.341;272154693170.355;272141085564.775;272127478648.777;272113872438.694;272100266933.600;272086662132.553;272073058052.041;272059454691.316;272045852049.635;272032250126.230;272018648920.326;272005048413.737;271991448588.102;271977849424.883;271964250922.778;271950653063.064;271937055844.239;271923459264.795;271909863323.215;271896268035.367;271882673417.285;271869079467.760;271855486185.557;271841893569.402;271828301617.991;271814710329.987;271801119721.433;271787529808.540;271773940607.686;271760352135.421;271746764391.069;271733177373.952;271719591083.378;271706005501.251;271692420609.278;271678836388.983;271665252839.086;271651669958.291;271638087745.292;271624506198.764;271610925299.966;271597345047.385;271583765439.495;271570186474.783;271556608151.750;271543030468.906;271529453424.753;271515877035.175;271502301316.215;271488726284.082;271475151955.154;271461578345.955;271448005455.798;271434433283.971;271420861829.732;271407291092.311;271393721070.920;271380151764.770;271366583155.672;271353015225.252;271339447972.315;271325881395.650;271312315494.038;271298750266.244;271285185711.023;271271621809.751;271258058560.990;271244495963.292;271230934015.189;271217372715.200;271203812061.829;271190252053.566;271176692706.241;271163134035.844;271149576058.513;271136018790.556;271122462248.467;271108906431.571;271095351339.212;271081796970.742;271068243325.532;271054690385.584;271041138132.704;271027586548.501;271014035614.376;271000485328.864;270986935690.476;270973386697.695;270959838348.993;270946290660.183;270932743629.901;270919197256.771;270905651539.402;270892106476.386;270878562066.307;270865018307.729;270851475199.204;270837932739.269;270824390943.779;270810849828.759;270797309410.391;270783769705.038;270770230729.222;270756692499.646;270743155015.852;270729618277.377;270716082266.418;270702546964.984;270689012354.902;270675478417.829;270661945135.248;270648412488.467;270634880458.620;270621349043.962;270607818242.731;270594288070.479;270580758542.901;270567229675.835;270553701485.263;270540173969.999;270526647128.842;270513120960.575;270499595463.969;270486070637.787;270472546480.776;270459022991.668;270445500169.186;270431978029.349;270418456588.340;270404935862.512;270391415868.389;270377896622.667;270364378124.908;270350860357.357;270337343302.073;270323826940.931;270310311255.612;270296796227.613;270283281838.240;270269768068.599;270256254916.894;270242742381.300;270229230477.289;270215719220.496;270202208626.743;270188698712.024;270175189475.220;270161680915.219;270148173030.888;270134665821.085;270121159284.656;270107653420.440;270094148227.260;270080643703.929;270067139849.247;270053636679.285;270040134210.280;270026632458.644;270013131440.952;269999631156.665;269986131605.234;269972632768.818;269959134629.373;269945637168.634;269932140368.128;269918644209.172;269905148672.883;269891653740.184;269878159409.087;269864665694.858;269851172612.922;269837680178.886;269824188408.534;269810697317.828;269797206905.634;269783717170.810;269770228112.200;269756739728.638;269743252018.957;269729764981.986;269716278616.563;269702792938.777;269689307964.880;269675823711.279;269662340194.536;269648857431.380;269635375404.184;269621894095.143;269608413486.262;269594933559.363;269581454296.084;269567975677.860;269554497685.942;269541020318.638;269527543574.233;269514067450.997;269500591964.448;269487117130.265;269473642947.040;269460169413.357;269446696527.775;269433224288.827;269419752695.011;269406281744.799;269392811436.637;269379341786.210;269365872809.360;269352404522.106;269338936940.633;269325470081.318;269312003943.480;269298538526.451;269285073829.559;269271609852.133;269258146593.489;269244684052.936;269231222212.542;269217761071.404;269204300628.623;269190840883.274;269177381834.410;269163923463.841;269150465753.165;269137008683.787;269123552236.916;269110096410.790;269096641203.629;269083186613.651;269069732639.052;269056279295.243;269042826597.805;269029374545.280;269015923136.204;269002472369.097;268989022242.465;268975572772.010;268962123973.600;268948675863.265;268935228439.987;268921781702.724;268908335650.405;268894890281.936;268881445613.412;268868001661.095;268854558441.421;268841115953.804;268827674197.659;268814233172.410;268800792860.265;268787353243.253;268773914320.413;268760476090.775;268747038553.357;268733601707.169;268720165534.019;268706730015.516;268693295133.088;268679860867.961;268666427201.146;268652994147.849;268639561723.424;268626129943.386;268612698806.219;268599268310.396;268585838454.376;268572409236.595;268558980672.666;268545552778.361;268532125569.616;268518699062.534;268505273256.210;268491848149.747;268478423742.253;268465000032.839;268451577020.619;268438154704.703;268424733084.207;268411312175.423;268397891977.620;268384472490.063;268371053712.012;268357635642.707;268344218264.197;268330801558.323;268317385506.718;268303970090.803;268290555291.798;268277141090.719;268263727502.725;268250314543.127;268236902227.396;268223490571.169;268210079573.093;268196669231.800;268183259545.911;268169850514.026;268156442134.742;268143034423.790;268129627397.068;268116221070.642;268102815460.744;268089410566.624;268076006387.544;268062602922.773;268049200171.587;268035798116.111;268022396738.300;268008996037.074;267995596011.333;267982196659.979;267968797981.891;267955399958.793;267942002572.196;267928605803.412;267915209650.701;267901814112.284;267888419203.483;267875024939.770;267861631336.781;267848238410.314;267834846159.195;267821454582.236;267808063678.239;267794673445.993;267781283884.274;267767895008.980;267754506836.177;267741119382.093;267727732645.999;267714346627.165;267700961307.724;267687576686.777;267674192763.435;267660809519.689;267647426937.355;267634044998.052;267620663700.331;267607283042.715;267593903023.689;267580523641.735;267567144895.298;267553766782.795;267540389302.618;267527012453.145;267513636249.852;267500260708.381;267486885844.537;267473511674.309;267460138196.746;267446765410.886;267433393315.758;267420021910.387;267406651193.779;267393281182.036;267379911874.293;267366543269.662;267353175367.232;267339808148.993;267326441596.760;267313075692.173;267299710416.705;267286345751.659;267272981695.267;267259618262.844;267246255469.865;267232893331.957;267219531864.896;267206171084.612;267192810990.084;267179451580.268;267166092854.100;267152734810.484;267139377448.312;267126020766.459;267112664763.793;267099309456.267;267085954842.910;267072600922.752;267059247694.808;267045895158.091;267032543294.509;267019192085.797;267005841513.482;266992491558.924;266979142220.385;266965793513.192;266952445452.831;266939098054.959;266925751318.325;266912405241.662;266899059823.696;266885715063.141;266872370975.786;266859027577.581;266845684884.644;266832342913.259;266819001662.802;266805661115.545;266792321253.565;266778982058.729;266765643512.713;266752305596.986;266738968309.892;266725631649.743;266712295631.919;266698960271.965;266685625568.536;266672291520.265;266658958125.786;266645625383.706;266632293309.684;266618961919.545;266605631229.297;266592301238.074;266578971945.009;266565643349.244;266552315432.845;266538988194.764;266525661633.945;266512335749.326;266499010539.832;266485685987.323;266472362073.452;266459038796.733;266445716155.654;266432394148.666;266419072774.198;266405752047.685;266392431984.711;266379112601.010;266365793912.469;266352475918.112;266339158616.965;266325842008.059;266312526073.370;266299210811.734;266285896221.989;266272582302.976;266259269036.505;266245956404.201;266232644404.553;266219333036.032;266206022314.122;266192712254.470;266179402872.891;266166094185.356;266152786190.979;266139478888.870;266126172278.109;266112866357.741;266099561109.743;266086256515.887;266072952574.767;266059649284.954;266046346645.007;266033044653.467;266019743325.893;266006442661.003;265993142657.513;265979843314.136;265966544629.595;265953246602.600;265939949231.852;265926652516.051;265913356470.893;265900061112.252;265886766439.153;265873472450.616;265860179145.636;265846886523.181;265833594565.185;265820303253.376;265807012586.313;265793722562.544;265780433180.599;265767144438.997;265753856336.236;265740568887.808;265727282109.356;265713996016.695;265700710625.799;265687425952.813;265674141980.035;265660858689.573;265647576063.349;265634294083.097;265621012730.379;265607731986.586;265594451866.920;265581172386.741;265567893561.564;265554615407.056;265541337939.048;265528061156.550;265514785058.566;265501509644.079;265488234912.077;265474960861.528;265461687474.396;265448414749.444;265435142685.413;265421871281.036;265408600535.017;265395330446.050;265382061012.832;265368792234.040;265355524108.329;265342256634.336;265328989810.666;265315723635.907;265302458125.600;265289193295.461;265275929161.377;265262665722.440;265249402960.774;265236140858.329;265222879396.864;265209618557.952;265196358356.920;265183098809.252;265169839930.594;265156581719.788;265143324175.648;265130067296.970;265116811099.499;265103555599.140;265090300778.046;265077046618.190;265063793118.311;265050540260.161;265037288042.251;265024036463.095;265010785538.141;264997535266.050;264984285645.465;264971036691.975;264957788421.350;264944540832.583;264931293924.680;264918047713.582;264904802198.447;264891557361.472;264878313184.668;264865069649.840;264851826738.597;264838584449.281;264825342797.185;264812101797.753;264798861466.593;264785621802.538;264772382804.406;264759144471.012;264745906801.150;264732669793.606;264719433464.090;264706197811.524;264692962834.824;264679728532.896;264666494904.630;264653261931.980;264640029613.658;264626797948.348;264613566934.717;264600336571.404;264587106857.037;264573877790.230;264560649386.515;264547421661.583;264534194614.349;264520968243.723;264507742548.621;264494517527.963;264481293163.752;264468069454.745;264454846399.687;264441623997.310;264428402246.337;264415181145.472;264401960693.413;264388740905.748;264375521781.313;264362303318.928;264349085517.393;264335868375.482;264322651908.870;264309436116.466;264296220997.159;264283006549.825;264269792773.337;264256579666.559;264243367211.425;264230155389.682;264216944182.892;264203733606.238;264190523675.055;264177314404.846;264164105794.361;264150897842.347;264137690564.460;264124483976.537;264111278094.586;264098072917.889;264084868428.817;264071664609.554;264058461459.018;264045258976.112;264032057142.811;264018855957.794;264005655419.718;263992455510.326;263979256211.175;263966057537.411;263952859504.337;263939662127.418;263926465405.382;263913269353.835;263900073971.638;263886879274.518;263873685278.352;263860491965.407;263847299317.769;263834107334.229;263820916013.590;263807725354.655;263794535356.251;263781346017.219;263768157319.529;263754969244.972;263741781792.014;263728594959.110;263715408761.580;263702223214.898;263689038334.685;263675854136.709;263662670636.891;263649487851.307;263636305762.468;263623124352.697;263609943604.143;263596763498.760;263583584035.195;263570405212.075;263557227027.993;263544049481.521;263530872571.199;263517696312.404;263504520720.687;263491345794.891;263478171533.850;263464997936.377;263451825001.275;263438652727.351;263425481113.426;263412310158.326;263399139860.878;263385970219.914;263372801234.260;263359632902.734;263346465224.138;263333298214.106;263320131871.582;263306966195.494;263293801184.743;263280636838.202;263267473154.728;263254310133.161;263241147772.340;263227986071.089;263214825028.225;263201664642.571;263188504912.954;263175345838.202;263162187400.301;263149029597.890;263135872429.587;263122715893.972;263109559989.587;263096404731.775;263083250136.028;263070096217.991;263056942993.472;263043790461.630;263030638621.623;263017487472.627;263004337013.805;262991187227.488;262978038095.822;262964889600.769;262951741724.100;262938594447.406;262925447768.907;262912301703.627;262899156266.743;262886011473.594;262872867339.675;262859723880.648;262846581112.328;262833439050.686;262820297695.044;262807157044.716;262794017099.014;262780877840.453;262767739251.373;262754601313.933;262741464026.924;262728327389.124;262715191399.300;262702056056.213;262688921358.601;262675787305.198;262662653894.714;262649521125.832;262636388997.190;262623257507.392;262610126655.023;262596996438.656;262583866856.865;262570737908.228;262557609591.314;262544481921.483;262531354914.254;262518228585.321;262505102950.548;262491978009.199;262478853760.536;262465730203.831;262452607338.333;262439485146.484;262426363610.521;262413242712.491;262400122434.260;262387002757.506;262373883680.507;262360765218.301;262347647386.090;262334530199.231;262321413673.240;262308297823.797;262295182666.739;262282068218.053;262268954477.102;262255841443.244;262242729099.061;262229617426.972;262216506409.201;262203396027.792;262190286281.366;262177177168.522;262164068687.850;262150960837.921;262137853617.302;262124747024.560;262111641058.262;262098535733.731;262085431066.445;262072327055.278;262059223699.083;262046120996.691;262033018946.929;262019917548.609;262006816800.532;261993716718.255;261980617317.506;261967518597.439;261954420557.208;261941323195.980;261928226512.913;261915130507.140;261902035177.774;261888940507.145;261875846477.394;261862753070.473;261849660268.154;261836568068.758;261823476487.354;261810385539.161;261797295239.560;261784205604.093;261771116648.457;261758028388.501;261744940840.223;261731853986.290;261718767809.202;261705682291.289;261692597414.713;261679513161.460;261666429530.084;261653346519.128;261640264127.127;261627182369.327;261614101261.133;261601020818.119;261587941056.011;261574861973.967;261561783571.135;261548705846.656;261535628782.933;261522552362.194;261509476583.213;261496401444.758;261483326945.580;261470253084.420;261457179860.008;261444107287.779;261431035383.324;261417964145.652;261404893573.754;261391823666.616;261378754423.202;261365685825.742;261352617872.991;261339550563.676;261326483896.516;261313417870.230;261300352483.538;261287287751.874;261274223690.840;261261160299.506;261248097576.932;261235035522.182;261221974117.619;261208913362.152;261195853254.688;261182793794.117;261169734979.305;261156676809.082;261143619282.245;261130562414.284;261117506220.836;261104450684.275;261091395786.793;261078341527.101;261065287903.898;261052234932.563;261039182628.644;261026130991.133;261013080019.025;261000029694.608;260986980016.702;260973930984.121;260960882595.698;260947834866.958;260934787796.900;260921741384.508;260908695628.765;260895650528.650;260882606083.132;260869562291.177;260856519151.733;260843476663.742;260830434826.124;260817393637.788;260804353097.635;260791313204.550;260778273957.417;260765235355.096;260752197396.417;260739160096.869;260726123472.101;260713087504.571;260700052176.556;260687017486.821;260673983450.796;260660950067.411;260647917335.578;260634885270.875;260621853872.376;260608823122.479;260595793003.397;260582763513.828;260569734669.121;260556706468.108;260543678909.603;260530651992.386;260517625731.878;260504600143.667;260491575243.498;260478551030.618;260465527487.605;260452504596.863;260439482340.611;260426460700.893;260413439676.243;260400419281.856;260387399533.112;260374380445.575;260361362034.983;260348344317.257;260335327275.189;260322310891.403;260309295148.343;260296280028.269;260283265529.916;260270251652.007;260257238409.885;260244225819.041;260231213895.112;260218202637.232;260205192044.519;260192182116.064;260179172834.306;260166164198.150;260153156206.483;260140148858.171;260127142152.042;260114136086.889;260101130678.101;260088125924.580;260075121825.216;260062118378.892;260049115584.475;260036113457.474;260023111996.942;260010111201.937;259997111071.529;259984111604.783;259971112784.148;259958114591.887;259945117010.086;259932120037.273;259919123688.604;259906127979.410;259893132925.196;259880138541.639;259867144844.574;259854151833.377;259841159490.807;259828167799.441;259815176758.302;259802186366.397;259789196622.707;259776207509.569;259763219009.132;259750231136.586;259737243907.284;259724257336.737;259711271424.004;259698286168.156;259685301568.274;259672317623.471;259659334332.876;259646351695.634;259633369710.885;259620388377.744;259607407678.689;259594427612.610;259581448178.359;259568469374.748;259555491217.169;259542513721.157;259529536902.395;259516560760.123;259503585293.562;259490610485.307;259477636317.755;259464662773.112;259451689833.395;259438717513.623;259425745828.993;259412774794.869;259399804426.790;259386834740.448;259373865751.709;259360897443.419;259347929798.254;259334962798.709;259321996427.119;259309030665.648;259296065512.882;259283100967.408;259270137044.385;259257173759.153;259244211127.215;259231249164.230;259218287886.034;259205327292.026;259192367381.593;259179408137.533;259166449542.468;259153491578.842;259140534245.493;259127577541.252;259114621464.937;259101666015.366;259088711207.917;259075757041.574;259062803515.323;259049850628.149;259036898395.605;259023946833.417;259010995957.472;258998045767.259;258985096245.685;258972147375.478;258959199139.184;258946251519.151;258933304514.109;258920358122.768;258907412360.389;258894467242.385;258881522784.333;258868579001.962;258855635894.606;258842693461.578;258829751685.626;258816810549.295;258803870051.491;258790930191.088;258777990966.945;258765052377.888;258752114422.711;258739177100.195;258726240409.124;258713304364.835;258700368982.843;258687434278.834;258674500268.670;258661566935.282;258648634261.420;258635702229.653;258622770822.368;258609840038.323;258596909876.256;258583980334.898;258571051412.976;258558123125.740;258545195488.623;258532268500.693;258519342161.019;258506416468.680;258493491422.748;258480567038.821;258467643332.645;258454720303.592;258441797951.015;258428876257.739;258415955206.409;258403034779.509;258390114975.888;258377195794.395;258364277233.883;258351359309.728;258338442020.942;258325525366.533;258312609345.496;258299693956.817;258286779215.995;258273865138.697;258260951740.758;258248039021.676;258235126980.965;258222215618.146;258209304916.231;258196394858.068;258183485426.315;258170576603.439;258157668371.714;258144760746.225;258131853742.187;258118947374.948;258106041659.985;258093136612.920;258080232249.521;258067328585.706;258054425604.542;258041523288.911;258028621621.517;258015720584.895;258002820161.414;257989920333.271;257977021115.493;257964122523.271;257951224571.947;257938327277.010;257925430654.103;257912534702.544;257899639421.643;257886744794.211;257873850802.876;257860957446.572;257848064724.228;257835172634.758;257822281193.571;257809390399.756;257796500252.397;257783610750.569;257770721893.328;257757833679.720;257744946108.767;257732059179.483;257719172890.874;257706287241.937;257693402231.657;257680517875.494;257667634172.604;257654751122.118;257641868723.184;257628986974.942;257616105876.550;257603225410.682;257590345576.326;257577466372.473;257564587798.133;257551709852.336;257538832550.584;257525955908.557;257513079925.644;257500204601.226;257487329934.679;257474455925.378;257461582556.215;257448709809.902;257435837685.419;257422966181.717;257410095314.193;257397225098.398;257384355550.044;257371486685.010;257358618486.421;257345750937.224;257332884020.200;257320017734.412;257307152078.930;257294287052.831;257281422638.742;257268558835.592;257255695658.771;257242833123.841;257229971230.070;257217109993.176;257204249412.576;257191389504.127;257178530267.381;257165671701.881;257152813807.157;257139956566.262;257127099962.045;257114243993.598;257101388659.976;257088533943.760;257075679827.336;257062826325.771;257049973454.283;257037121228.250;257024269663.206;257011418758.416;256998568513.129;256985718926.588;256972869981.588;256960021677.184;256947174012.424;256934326986.345;256921480597.977;256908634846.341;256895789730.435;256882945249.249;256870101401.769;256857258186.964;256844415603.801;256831573667.655;256818732377.651;256805891732.915;256793051732.569;256780212375.744;256767373661.557;256754535605.531;256741698206.924;256728861464.985;256716025378.973;256703189948.151;256690355155.383;256677520983.355;256664687430.998;256651854497.235;256639022180.972;256626190481.108;256613359412.938;256600528991.913;256587699233.648;256574870137.506;256562041702.852;256549213929.036;256536386815.423;256523560344.998;256510734500.596;256497909281.306;256485084686.213;256472260714.387;256459437364.883;256446614653.142;256433792594.761;256420971205.508;256408150501.309;256395330481.858;256382511130.461;256369692430.246;256356874364.165;256344056915.004;256331240065.358;256318423814.036;256305608159.837;256292793117.971;256279978703.818;256267164932.942;256254351804.689;256241539318.406;256228727473.443;256215916269.143;256203105721.221;256190295845.559;256177486658.199;256164678142.595;256151870282.027;256139063059.605;256126256474.638;256113450526.431;256100645214.300;256087840537.568;256075036479.194;256062233038.351;256049430214.195;256036628005.894;256023826412.610;256011025449.879;255998225133.384;255985425478.963;255972626502.599;255959828220.417;255947030615.968;255934233672.618;255921437373.544;255908641701.746;255895846640.047;255883052187.447;255870258342.940;255857465105.500;255844672490.447;255831880513.240;255819089189.493;255806298518.610;255793508499.978;255780719132.966;255767930416.927;255755142351.212;255742354935.161;255729568151.750;255716782000.118;255703996479.381;255691211588.644;255678427326.986;255665643693.480;255652860687.170;255640078323.427;255627296617.774;255614515569.540;255601735178.029;255588955442.529;255576176362.310;255563397936.615;255550620148.326;255537842980.132;255525066414.550;255512290450.278;255499515086.026;255486740320.526;255473966168.851;255461192646.240;255448419768.095;255435647549.970;255422876007.566;255410105156.735;255397334997.155;255384565528.486;255371796734.046;255359028596.988;255346261100.305;255333494226.825;255320727959.210;255307962279.938;255295197187.628;255282432680.882;255269668774.618;255256905483.916;255244142824.027;255231380810.384;255218619458.603;255205858784.501;255193098787.752;255180339468.029;255167580825.008;255154822858.352;255142065551.414;255129308887.370;255107066034.410;255084825100.143;255062586085.559;255040348991.674;255018113819.521;254995880570.154;254973649244.636;254951419844.052;254929192369.505;254916446191.415;254903700629.969;254890955683.949;254878211352.128;254865467633.266;254852724526.112;254839982045.688;254827240207.175;254814499025.922;254801758517.449;254789018697.450;254776279565.510;254763541121.211;254741330150.813;254719121144.005;254696914085.849;254674708961.270;254652505755.046;254630304451.806;254608105036.013;254585907491.964;254573178437.979;254560449980.115;254547722133.289;254534994912.577;254522268333.227;254509542410.658;254496817160.462;254484092598.388;254471368724.093;254458645537.244;254445923021.249;254433201159.354;254420479934.644;254398298869.641;254376119698.447;254353942421.821;254331767040.530;254309593555.373;254287421967.166;254265252293.009;254243084550.179;254230372615.622;254217661324.018;254204950674.676;254192240666.913;254179531300.054;254166822573.430;254154114486.366;254141407038.164;254128700244.357;254115994104.372;254103288617.640;254090583783.579;254068431411.336;254046280981.462;254024132495.337;254001985954.358;253979841359.946;253957698697.311;253935557951.531;253922861541.477;253910165745.403;253897470562.414;253884775991.602;253872082032.069;253859388682.923;253846695943.272;253834003828.425;253821312353.842;253808621535.129;253795931371.811;253783241863.405;253770553009.419;253748428513.909;253726305959.578;253704185347.875;253682066664.038;253659949893.141;253637835036.311;253615722094.685;253593611069.395;253580931671.009;253568252886.867;253555574732.173;253542897222.279;253530220372.681;253517544182.813;253504868652.088;253492193779.901;253479519565.627;253466845992.420;253454173043.249;253441500700.887;253428828964.134;253416157831.778;253394064183.928;253371972424.004;253349882568.945;253327794635.869;253305708642.074;253283624588.842;253261542477.467;253239462309.260;253226800626.469;253214139586.699;253201479189.093;253188819432.780;253176160316.876;253163501840.487;253150844002.705;253138186802.599;253125530239.204;253112874295.339;253100218969.790;253087564261.301;253074910168.585;253062256690.330;253040193886.638;253018133011.957;252996074083.439;252974017102.250;252951962069.569;252929908970.422;252907857789.697;252895212722.572;252882568259.852;252869924416.298;252857281206.832;252844638646.508;252831996750.527;252819355501.907;252806714883.481;252794074877.887;252781435467.572;252768796667.109;252756158491.232;252734122379.417;252712088206.447;252690055973.436;252668025681.506;252645997315.632;252623970860.627;252601946317.292;252589316515.891;252576687328.452;252564058769.728;252551430854.622;252538803582.051;252526176950.914;252513550960.103;252500925608.506;252488300895.027;252475676818.590;252463053378.133;252450430556.467;252437808336.230;252425186732.172;252412565759.214;252390559618.444;252368555387.266;252346553082.799;252324552722.338;252302554291.089;252280557774.114;252258563172.464;252236570487.221;252223958966.351;252211348066.291;252198737802.063;252186128188.846;252173519225.864;252160910896.207;252148303182.791;252135696068.351;252123089551.561;252110483647.198;252097878370.200;252085273735.653;252072669758.804;252060066455.059;252047463807.761;252025489582.052;252003517258.432;251981546821.807;251959578256.940;251937611580.638;251915646809.897;251893683961.877;251881089550.094;251868495801.246;251855902714.708;251843310273.751;251830718461.490;251818127276.963;251805536719.221;251792946787.314;251780357464.209;251767768748.811;251755180640.020;251742593152.837;251730006302.424;251708059689.191;251686115022.579;251664172288.228;251642231471.635;251620292558.149;251598355532.978;251585775913.415;251573196892.352;251560618484.785;251548040705.863;251535463570.899;251522887079.290;251510311230.421;251497736023.685;251485161458.455;251472587534.089;251460014249.923;251438091230.393;251416170128.652;251394250930.054;251372333619.818;251350418183.029;251337850932.272;251325284299.394;251312718299.620;251300152948.349;251287588245.089;251275024189.346;251262460780.617;251249898002.334;251237335837.751;251224774286.008;251202871717.365;251180971037.995;251159072249.140;251137175368.105;251115280412.351;251102724950.580;251090170144.307;251077615993.102;251065062496.543;251052509638.165;251039957401.354;251027405769.326;251014854725.146;251002304267.747;250989754396.072;250977205125.112;250955324046.886;250933444876.189;250911567630.670;250889692328.189;250867818986.815;250855275961.112;250842733597.596;250830191880.093;250817650792.256;250805110317.575;250792570439.366;250780031156.830;250767492469.175;250754954375.599;250742416875.299;250720556289.026;250698697602.463;250676840833.282;250654985983.315;250633133054.416;250620601672.093;250608070927.634;250595540820.684;250583011350.884;250570482517.877;250557954321.305;250545426760.816;250532899836.051;250520373546.650;250507847892.244;250495322872.470;250482798486.945;250460960676.252;250439124739.951;250417290679.417;250395458496.023;250373628191.152;250351799766.183;250339282387.102;250326765620.202;250314249480.789;250301733984.335;250289219146.483;250276704983.068;250264191494.129;250251678679.731;250239166523.971;250226655010.806;250214144124.056;250201633847.401;250189124164.374;250167311940.869;250145501565.477;250123693039.625;250101886380.758;250080081606.526;250067577796.318;250055074623.637;250042572088.464;250030070190.780;250017568930.551;250005068307.735;249992568322.269;249980068974.082;249967570263.086;249945777309.718;249923986266.363;249902197135.121;249880409918.116;249867916250.417;249855423219.331;249842930824.776;249830439066.672;249817947928.964;249805457395.429;249792967465.638;249780478139.145;249767989415.512;249755501294.291;249733726778.942;249711954135.805;249690183382.565;249668414537.096;249646647617.458;249634165567.514;249621684166.030;249609203413.023;249596723308.495;249584243852.439;249571765028.882;249559286821.691;249546809214.584;249534332207.055;249521855798.614;249509379988.770;249496904777.049;249475152783.950;249453402672.722;249431654461.208;249409908167.444;249388163793.748;249375694674.556;249363226200.231;249350758370.925;249338291186.826;249325824632.184;249313358691.109;249300893363.476;249288428649.174;249275964548.101;249263501060.174;249241769487.409;249220039812.591;249198312038.047;249176586166.135;249164127661.182;249151669785.624;249139212539.601;249126755907.328;249114299872.852;249101844435.975;249089389596.504;249076935354.254;249064481724.964;249042767426.977;249021055042.104;248999344588.770;248977636069.680;248965187535.045;248952739630.976;248940292341.747;248927845667.376;248915399607.889;248902954163.308;248890509333.659;248878065118.965;248865621519.231;248853178534.460;248831482721.507;248809788789.790;248788096741.520;248766406578.922;248744718320.164;248732281385.878;248719845098.868;248707409459.525;248694974468.253;248682540125.444;248670106415.593;248657673323.014;248645240831.847;248632808926.064;248620377589.463;248607946821.553;248595516637.728;248583087053.540;248561414685.395;248539744212.046;248518075651.711;248496409022.793;248474744327.982;248453081569.987;248440659292.544;248428237648.395;248415816621.788;248403396196.795;248390976373.190;248378557150.732;248366138529.158;248353720508.200;248341303087.564;248328886282.798;248316470109.586;248304054567.875;248291639657.589;248279225378.626;248266811730.873;248245167075.841;248223524312.750;248201883443.643;248180244454.741;248158607347.989;248136972125.358;248124565596.532;248112159681.268;248099754379.272;248087349690.255;248074945613.925;248062542149.984;248050139298.105;248037737057.957;248025335429.211;248012934411.528;248000534004.570;247988134207.994;247975735021.454;247954115642.363;247932498165.063;247910882607.523;247889268972.063;247867657261.016;247846047476.725;247833655592.244;247821264333.262;247808873683.700;247796483627.311;247784094147.687;247771705228.241;247759316883.869;247746929129.618;247734541980.695;247722155452.462;247709769560.444;247697384304.493;247684999684.456;247672615700.176;247660232351.477;247638640556.644;247617050655.236;247595462649.243;247573876540.687;247552292331.612;247530710024.072;247518333880.988;247505958355.984;247493583432.836;247481209095.136;247468835342.081;247456462172.848;247444089586.592;247431717582.435;247419346175.284;247406975380.179;247394605212.283;247382235686.898;247369866819.473;247357498625.607;247335933277.040;247314369842.932;247292808309.411;247271248662.456;247249690887.881;247228134987.159;247215773974.262;247203413567.813;247191053766.973;247178694570.886;247166335962.866;247153977941.811;247141620506.583;247129263656.011;247116907388.880;247104551719.739;247092196663.256;247079842234.232;247067488447.613;247055135302.691;247033596212.592;247012059018.180;246990523736.655;246968990369.623;246947458918.695;246925929369.706;246913583483.468;246901238201.320;246888893506.292;246876549397.004;246864205872.050;246851862929.996;246839520585.115;246827178851.804;246814837728.802;246802497214.821;246790157308.540;246777818008.622;246765479313.718;246753141238.260;246731628443.751;246710117552.201;246688608580.701;246667101546.531;246645596435.665;246624093233.950;246611762473.815;246599432326.464;246587102775.036;246574773802.500;246562445407.433;246550117588.404;246537790343.987;246525463672.742;246513137588.939;246500812106.992;246488487241.468;246476162991.329;246463839355.534;246451516333.045;246439193922.824;246417708444.612;246396224828.309;246374743075.045;246353263185.973;246331785178.000;246310309068.204;246297993758.195;246285679115.948;246273365141.089;246261051833.236;246248739192.008;246236427217.003;246224115892.053;246211805200.792;246199495126.659;246187185652.896;246174876762.551;246162568438.480;246141107271.682;246119647904.300;246098190336.648;246076734584.764;246064430864.377;246052127731.035;246039825198.971;246027523282.541;246015221996.248;246002921354.726;245990621372.751;245969175049.957;245947730638.880;245926288140.860;245913992160.598;245901696835.529;245889402149.025;245877108084.258;245864814624.202;245852521767.349;245840229512.159;245827937857.068;245806505848.153;245785075688.052;245763647377.292;245742220916.389;245729934065.003;245717647803.520;245705362130.116;245693077042.939;245680792540.100;245668508619.681;245656225279.724;245643942533.922;245631660396.092;245619378880.313;245597964805.048;245576552630.847;245555142374.502;245533734052.967;245521457676.083;245509181977.817;245496906941.692;245484632551.047;245472358789.039;245460085638.634;245447813082.606;245435541103.537;245414143340.456;245392747400.880;245371353284.956;245359084962.578;245346817222.580;245334550078.613;245322283544.462;245310017618.357;245297752298.506;245276366348.183;245254982251.164;245242719651.187;245230457650.064;245218196245.910;245196817160.415;245184557416.038;245172298294.311;245150923239.499;245138665861.035;245126409147.613;245114153113.650;245092783349.525;245071415497.955;245059162360.234;245046909850.188;245034657950.567;245022406643.945;245010155912.715;244997905739.086;244976546034.264;244955188131.199;244933832045.289;244921585491.878;244909339531.392;244897094177.437;244884849443.760;244872605328.601;244860361830.186;244848118946.717;244826772161.716;244805427235.927;244793187096.717;244780947565.038;244759606621.657;244747368754.448;244735131504.652;244722894886.085;244701559054.046;244689324194.845;244677089993.425;244655758292.277;244643525856.971;244631294075.024;244609966491.886;244597736451.257;244585507027.940;244573278204.410;244551955666.486;244530634940.894;244518408734.651;244506183087.484;244493957996.804;244481733460.000;244469509490.055;244457286100.089;244435973319.619;244414662391.840;244393353332.445;244381133759.643;244368914834.398;244356696555.021;244344478919.796;244332261926.988;244320045574.852;244307829861.618;244286530360.816;244265232724.800;244253019795.094;244249890213.069;244246760679.891;244243631191.651;244240501744.386;244237372334.072;244234242956.630;244231113607.872;244218902372.645;244197610612.138;244176320695.110;244164112154.841;244151904206.885;244139696848.919;244127490078.600;244115283893.559;244103078307.000;244081796519.609;244060516601.907;244039238569.634;244027036872.973;244014835827.150;244011709297.734;244008582836.421;244005456423.786;244002330040.203;243999203665.841;243996077296.194;243983877754.015;243971678775.168;243950408453.973;243929139977.162;243916943701.319;243904748049.548;243892553035.366;243880358656.835;243868164911.991;243855971798.848;243843779315.397;243831587444.087;243819396182.761;243798139266.070;243776884196.475;243764695659.058;243752507731.488;243749384520.123;243746261338.259;243743138181.448;243740015045.180;243736891924.871;243733768831.457;243721582361.339;243709396519.470;243697211319.207;243675965054.874;243654720668.884;243633478145.681;243612237469.540;243600057179.922;243587877471.468;243575698341.370;243563519802.339;243551341867.187;243548221181.152;243545100561.134;243541980018.312;243538859548.406;243526682962.341;243514506975.510;243502331569.826;243481102277.109;243468928493.160;243456755302.592;243444582718.070;243432410752.358;243420239418.332;243408068713.426;243395898635.037;243383729165.042;243362510221.419;243341293104.633;243320077829.308;243298864410.225;243295747012.904;243292629676.095;243289512395.140;243286395165.328;243283277981.900;243280160824.555;243277043672.759;243273926536.780;243261763111.751;243249600301.945;243228393044.101;243207187649.251;243185984101.303;243164782399.478;243143582542.992;243131425658.257;243119269368.828;243107113687.227;243094958626.111;243082804198.266;243070650385.602;243058497169.849;243055382844.281;243052268548.209;243049154292.088;243046040086.465;243042925926.455;243030774077.239;243018622826.911;243006472172.564;242985286092.790;242964101868.979;242942919500.443;242930772654.624;242918626414.223;242906480776.258;242894335737.719;242882191295.567;242870047446.726;242857904203.612;242845761578.771;242833619584.874;242821478234.727;242809337510.303;242806226435.777;242803115389.127;242790975540.411;242778836271.596;242766697595.032;242754559523.178;242742422053.105;242721258966.371;242709123146.208;242696987921.032;242684853303.164;242672719305.053;242660585939.275;242648453218.540;242636321140.243;242624189686.257;242612058838.260;242599928577.738;242587798901.406;242584690612.891;242581582340.064;242569453506.353;242557325254.647;242545197597.118;242533070546.046;242520944113.823;242499800338.636;242487675644.627;242475551611.256;242463428220.436;242451305453.878;242439183293.091;242427061719.375;242414940713.831;242402820272.761;242390700392.403;242378581068.933;242366462298.461;242354344092.512;242342226462.710;242321098079.456;242308982080.063;242296866695.914;242284751939.115;242272637821.880;242260524356.525;242248411555.481;242245307593.382;242242203713.328;242230091961.947;242217980858.404;242205870384.071;242193760520.113;242181651247.486;242169542546.936;242157434398.996;242145326799.392;242133219759.261;242121113289.837;242109007402.470;242096902108.607;242084797419.800;242072693347.714;242051588587.876;242039486224.728;242027384515.518;242015283456.934;242003183045.611;241991083278.144;241978984135.682;241966885599.159;241954787647.088;241942690260.270;241930593434.346;241918497164.893;241906401447.418;241894306292.806;241882211712.029;241870117716.148;241858024316.315;241845931523.767;241833839349.837;241821747805.946;241809656903.608;241806558522.063;241803460245.488;241800362068.125;241797263984.135;241785174567.209;241773085758.651;241752007368.697;241730930789.191;241709856017.568;241688783051.227;241667711887.519;241646642523.762;241634560421.491;241622478885.693;241610397926.974;241598317556.032;241586237783.666;241583142179.845;241580046635.796;241576951160.488;241573855762.978;241570760452.392;241558682243.000;241546604664.427;241534527697.026;241522451320.933;241510375531.424;241498300323.711;241477245738.745;241456192951.193;241435141958.086;241414092756.412;241402022183.624;241389952198.831;241377882812.571;241365814035.489;241362721231.104;241350653240.012;241338585885.082;241326519177.282;241314453112.251;241302387685.556;241290322892.699;241278258729.103;241266195190.119;241254132255.679;241242069905.481;241230008118.989;241226917204.401;241223826298.358;241220735408.665;241217644543.185;241214553709.836;241193523777.175;241172495686.583;241151469435.048;241130445019.508;241109422436.869;241088401683.982;241067382757.660;241046365654.673;241034313535.601;241022262024.494;241019173721.614;241016085487.262;241012997329.653;241009909241.738;241006821216.381;240994771221.973;240982721831.057;240970673038.291;240958624838.264;240946577225.501;240934530194.454;240922483739.502;240910437854.959;240889434447.832;240868432853.092;240847433082.518;240826435147.993;240814394038.810;240802353565.708;240790313738.840;240778274553.145;240775189498.973;240772104486.431;240760066224.827;240748028560.772;240735991488.641;240723955002.738;240711919097.286;240699883766.427;240687849019.578;240675814866.231;240663781300.650;240651748317.011;240639715924.729;240627684133.293;240615652952.292;240612569858.398;240609486836.558;240597456635.376;240585427045.302;240573398060.516;240552424113.281;240531451993.689;240510481697.887;240489513221.973;240468546562.006;240447581714.009;240435559570.400;240423538022.962;240411517081.113;240408436615.465;240405356184.299;240402275794.816;240399195454.289;240387175757.234;240375156692.440;240363138254.361;240351120437.394;240339103235.880;240327086644.094;240315070656.262;240303055266.544;240291040469.042;240279026242.518;240275947517.288;240263933955.713;240251920954.735;240239908523.294;240227896670.421;240206952693.558;240186010547.231;240165070242.970;240144131792.432;240123195207.390;240111189303.139;240099184021.033;240087179355.435;240075175300.654;240063171850.940;240051169000.481;240039166743.406;240036091104.950;240033015491.112;240029939893.647;240026864319.537;240023788775.840;240011787813.712;239999787464.482;239987787722.458;239975788581.881;239963790036.922;239942869200.067;239921950194.255;239901033015.627;239880117660.278;239859204124.250;239847211408.903;239835219283.689;239823227742.383;239811236778.679;239799246383.348;239787256547.534;239775267264.763;239763278543.790;239751290393.438;239739302822.615;239727315840.309;239715329455.590;239703343677.629;239700272056.991;239697200504.051;239694129026.060;239691057615.049;239679073183.229;239667089368.965;239655106166.342;239643123569.387;239631141572.064;239610249640.620;239589359510.630;239568471177.788;239547584637.739;239526699886.087;239514723577.792;239502747860.860;239490772744.248;239478798236.997;239466824348.247;239454851083.290;239442878450.757;239430906460.179;239418935121.183;239406964443.510;239394994421.740;239383025050.413;239371056308.803;239359088175.983;239347120630.823;239335153667.202;239323187278.948;239311221459.841;239308155165.592;239305088873.235;239302022574.195;239298956259.815;239295889936.628;239292823611.248;239280859020.508;239259997508.013;239239137795.379;239218279893.806;239197423814.623;239176569569.288;239155717169.396;239134866611.416;239114017891.804;239093171006.984;239072325953.365;239051482727.334;239039530288.068;239027578435.866;239015627164.895;239012564438.903;239009501729.229;239006439027.651;239003376325.877;239000313630.761;238997250949.233;238985301004.561;238973351654.299;238961402907.937;238949454775.063;238937507265.362;238925560388.623;238913614154.737;238901668573.695;238889723640.361;238877779349.547;238865835696.009;238853892674.452;238833068655.510;238821127319.971;238809186596.181;238797246478.413;238785306960.876;238773368037.722;238761429703.046;238740613769.695;238719799640.340;238698987311.171;238678176778.354;238666243126.589;238654310033.054;238642377491.774;238630445511.903;238618514102.699;238606583273.531;238594653033.881;238582723393.339;238570794361.608;238558865948.506;238546938163.967;238543881542.419;238540824986.821;238537768489.746;238534712043.686;238531655641.078;238528599274.286;238516673082.282;238504747476.556;238483953807.599;238463161921.662;238442371815.027;238421583483.953;238400796924.672;238380012133.402;238359229106.334;238338447854.834;238317668390.395;238296890724.646;238284975780.673;238273061431.421;238270008045.589;238266954722.688;238263901470.718;238260848297.772;238257795212.046;238254742206.643;238242829602.260;238230917629.641;238219006284.043;238207095560.683;238195185454.742;238183275961.365;238171367075.660;238159458792.695;238147551107.505;238126788654.883;238106027997.222;238085269131.301;238064512053.883;238043756761.712;238031854755.345;238019953333.144;238008052505.020;237996152281.000;237993102725.430;237990053212.222;237978153904.656;237966255190.281;237954357079.384;237942459582.369;237930562709.769;237918666457.072;237906770819.735;237894875793.174;237882981372.782;237871087553.902;237859194331.834;237847301701.824;237835409659.074;237823518198.735;237811627312.059;237799736994.837;237787847241.837;237775958047.764;237764069407.271;237752181314.953;237731453114.573;237710726673.484;237690002003.499;237678117460.780;237666233489.559;237654350099.975;237642467302.273;237630585106.805;237618703524.028;237606822564.508;237594942238.924;237583062558.066;237571183517.697;237559305113.539;237547427341.264;237544383569.731;237541339871.664;237538296240.431;237526419636.843;237514543635.026;237502668230.103;237481962039.034;237461257654.816;237440555074.630;237419854295.627;237399155314.928;237387285654.140;237375416564.778;237363548041.880;237351680080.449;237339812690.567;237327945882.442;237316079666.396;237313038821.878;237309998024.303;237306957266.920;237303916542.915;237292051544.219;237280187155.503;237268323387.465;237247637567.817;237226953564.135;237206271373.914;237185590994.646;237173732014.099;237161873630.521;237150015839.250;237129340371.992;237117484198.108;237105628600.498;237093773574.547;237081919115.608;237070065218.997;237058211879.997;237046359093.852;237043321598.569;237040284094.799;237037246590.574;237034209094.020;237022357327.981;237010506146.922;236989842483.314;236969180629.901;236948520599.495;236927862405.063;236907206044.630;236886551516.208;236865898817.812;236854055643.943;236851020669.537;236847985749.414;236844950877.359;236841916047.101;236830074115.863;236818232773.264;236806392015.158;236794551837.363;236782712235.672;236770873205.845;236759034758.692;236738393174.039;236726556357.697;236714720155.207;236702884577.645;236682247997.662;236661613262.805;236640980371.471;236629148659.155;236617317571.264;236605487104.211;236602455585.304;236599424097.684;236587594500.719;236575765463.970;236563936983.245;236560905854.941;236549078033.734;236537250770.874;236525424076.069;236504802915.846;236484183551.311;236463565995.620;236451742989.189;236439920581.342;236428098767.371;236416277541.358;236404456899.556;236392636838.206;236380817353.509;236368998441.652;236365969651.848;236362940875.959;236359912107.893;236348094145.363;236327488315.666;236306884282.481;236286282059.160;236265681644.163;236245083035.949;236224486232.974;236221459260.553;236218432326.598;236206621473.409;236194811202.409;236191784584.194;236179975042.750;236168166102.994;236156357776.643;236153331704.706;236141524156.528;236120936442.043;236109130529.812;236088545650.631;236067962569.757;236056159316.941;236035579063.037;236023777443.358;236011976434.393;236008952324.480;236005928282.354;235994128254.566;235982328840.314;235970530036.490;235958731839.967;235946934232.570;235935137195.946;235923340724.267;235911544828.115;235899749503.772;235879182974.052;235858618217.718;235838055233.084;235826263513.697;235814472381.790;235802681833.990;235790891866.900;235787870531.117;235784849219.525;235781827926.445;235770038932.354;235758250501.493;235746462645.251;235725909294.943;235705357740.805;235684807981.409;235664260015.300;235652476861.397;235640694297.141;235637674787.374;235625892946.485;235614111679.790;235602330983.789;235590550869.947;235578771349.850;235566992435.199;235563973852.473;235560955319.346;235549177325.928;235528641181.282;235508106826.384;235487574259.745;235467043479.865;235455270167.641;235443497431.116;235431725281.799;235419953731.323;235416937054.697;235413920452.716;235410903920.245;235407887452.101;235387363746.611;235366841849.524;235346321759.544;235325803475.364;235305286995.667;235293521985.593;235281757570.005;235269993745.635;235258230509.183;235255216119.479;235252201778.797;235249187481.761;235246173222.941;235234411202.492;235213902725.606;235193396024.640;235172891098.044;235152387944.258;235131886561.722;235111386948.876;235099631564.087;235087876738.682;235084864381.005;235081852028.878;235078839691.575;235075827378.483;235064073639.289;235052320499.024;235040567969.643;235020076220.403;234999586272.088;234979098123.602;234958611773.851;234938127221.741;234926380481.275;234923370239.116;234920360038.117;234917349873.142;234914339739.010;234902594173.404;234890849183.158;234879104765.197;234858627089.915;234838151175.649;234817677021.070;234797204624.838;234776734000.561;234773725521.378;234770717090.942;234767708719.297;234764700416.605;234752962288.891;234741224776.190;234729487875.998;234709023324.449;234688560575.670;234668099628.921;234647640483.458;234635908353.080;234624176819.830;234612445881.115;234609439767.852;234606433698.961;234603427669.688;234591697792.400;234579968506.514;234559517114.141;234547789451.784;234536062378.310;234524335891.149;234503889355.543;234483444597.890;234471720726.138;234459997430.264;234448274707.634;234436552555.602;234416113603.952;234395676412.847;234392673003.761;234389669611.278;234377950281.965;234374947019.852;234371943759.752;234360225220.861;234339792725.281;234328075784.650;234316359445.082;234304643704.377;234284216097.767;234272501983.767;234252077194.165;234240364697.849;234228652789.205;234208231828.331;234187812645.982;234176103357.641;234164394647.819;234161394086.425;234158393556.344;234155393053.160;234143685332.206;234131978189.209;234120271636.567;234099860085.165;234088155199.639;234076450932.679;234064747282.523;234053044247.396;234041341825.520;234029640015.112;234009236575.775;233988834936.450;233968435097.105;233948037057.732;233936340017.761;233924643578.350;233921646394.246;233918649265.219;233915652172.489;233912655097.089;233909658019.864;233897962834.711;233877570846.134;233857180589.239;233836792078.332;233816405327.881;233804714657.674;233793024577.697;233781335086.172;233769646181.322;233757957861.361;233746270124.495;233734582968.928;233722896392.858;233702519561.651;233690834574.679;233679150165.404;233667466331.795;233655783071.805;233652789040.802;233649795027.181;233646801026.927;233635118715.384;233623436965.021;233611755773.963;233600075155.194;233579708855.085;233559344342.772;233538981618.572;233518620682.832;233498261535.908;233486586652.692;233474912360.907;233463238659.118;233451565545.887;233439893019.774;233428221079.339;233416549723.135;233413558715.630;233401888099.434;233390218064.347;233378548608.682;233366879745.599;233355211488.399;233352221313.431;233349231190.036;233337563861.124;233325897133.056;233314231004.688;233302565474.882;233290900542.495;233279236206.387;233258898126.686;233238561842.011;233218227367.935;233197894705.374;233177563840.440;233157234759.119;233145577243.956;233133920301.854;233122263931.535;233110608131.722;233098952901.139;233095966190.566;233092979508.900;233089992852.820;233087006218.995;233075352091.450;233063698519.982;233052045503.021;233040393038.988;233020075679.512;232999760083.748;232988110208.434;232976460917.125;232964812208.851;232953164082.648;232941516537.554;232929869572.605;232918223186.847;232906577379.326;232886271672.140;232865967726.612;232845665543.527;232825365123.683;232805066482.707;232793426333.805;232790443377.647;232787460485.013;232784477653.276;232781494879.784;232778512161.879;232775529496.882;232772546882.101;232760908497.484;232740615646.081;232720324576.663;232700035290.394;232679747788.468;232659462072.103;232647829426.967;232636197375.477;232624565917.118;232612935051.375;232601304762.956;232589675036.421;232578045870.972;232566417265.811;232554789220.150;232534514335.733;232514241206.337;232493969847.822;232482345410.859;232470721583.398;232459098365.002;232456119911.994;232453141516.975;232450163177.622;232447184891.598;232444206656.557;232432584865.787;232420963673.938;232409343080.803;232389081098.325;232368820896.592;232348562477.125;232328305826.700;232308050946.742;232296436057.169;232284821749.814;232273208024.401;232261594880.661;232249982318.333;232238370337.171;232226758936.939;232215148121.589;232203537889.826;232191928241.239;232180319175.421;232168710691.968;232165736150.518;232162761642.014;232159787149.430;232148179644.044;232136572702.787;232124966325.383;232113360511.567;232093124275.925;232072889784.633;232052657039.147;232032426040.960;232012196791.583;232000596511.362;231988996793.044;231977397636.443;231965799041.370;231954201007.628;231942603549.774;231931006682.512;231910786217.613;231899190994.225;231887596368.441;231876002340.195;231864408909.417;231861438118.259;231858467376.296;231855496681.600;231852526032.220;231849555426.186;231837963371.615;231826371909.526;231806160730.059;231785951323.359;231765743691.253;231745537835.586;231733951061.600;231722364880.850;231710779293.479;231699194299.637;231678994303.592;231658796067.957;231647213672.416;231635631855.778;231615437400.312;231595244705.920;231583665487.210;231572086847.358;231569119963.712;231566153125.635;231554575376.601;231542998209.049;231531421622.758;231528455278.014;231525488968.790;231522522693.009;231519556448.581;231499372098.804;231479189493.369;231459008633.813;231447436168.615;231435864267.047;231424292928.957;231412722154.187;231401151942.584;231380977793.277;231360805385.300;231340634720.171;231320465799.425;231308900121.756;231297335006.201;231285770452.599;231274206460.782;231262643030.581;231251080178.325;231239517922.894;231227956264.229;231224993578.910;231222030945.897;231219068363.308;231216105829.239;231204545402.438;231192985559.269;231172829548.291;231152675311.379;231141118104.818;231129561490.744;231118005469.313;231106450040.686;231094895205.024;231083340962.491;231071787313.253;231051641990.235;231031498421.612;231011356609.061;230991216554.272;230971078258.959;230959530229.978;230947982779.770;230936435908.349;230924889615.738;230921931012.652;230918972457.056;230907427057.768;230895882239.494;230884338002.023;230872794345.143;230869836437.884;230866878566.317;230863920728.402;230860962922.083;230849420422.781;230829294504.482;230809170326.695;230789047890.948;230768927198.786;230757389237.340;230745851839.726;230734315005.810;230722778735.463;230711243028.550;230699707884.942;230679594888.778;230659483630.188;230639374110.692;230619266331.823;230607735717.972;230596205666.519;230584676162.662;230573147206.106;230561618811.211;230550090991.325;230538563769.724;230527037146.385;230515511121.290;230512557578.378;230509604089.847;230506650653.874;230503697268.610;230500743932.191;230489219306.037;230477695261.995;230457601647.361;230446079215.808;230434557377.224;230423036131.794;230402947351.365;230382860325.201;230371341687.624;230359823628.961;230348306149.252;230328223898.231;230308143400.811;230288064658.683;230276550779.768;230265037479.954;230253524759.282;230242012617.800;230230501055.556;230210429102.560;230190358901.439;230170290453.869;230150223761.541;230147275095.420;230144326466.189;230141377871.847;230138429310.377;230135480779.742;230123974523.450;230112468829.874;230100963698.638;230089459129.365;230086511160.111;230083563209.002;230072059452.133;230060556258.903;230049053629.205;230037551562.921;230017496264.914;229997442701.047;229977390872.828;229957340781.789;229937292429.471;229925795864.493;229914299847.253;229902804392.094;229891309513.517;229879815226.162;229868321530.215;229856828425.862;229836788838.540;229816751012.550;229796714949.691;229785225493.260;229773736639.938;229770792690.819;229767848802.601;229764904973.558;229761961187.344;229759017442.033;229747529981.361;229736043093.210;229724556777.324;229713071031.949;229701585865.736;229690101278.755;229678617271.068;229658593393.780;229638571266.217;229618550890.070;229598532267.047;229587052864.487;229575574041.542;229564095798.284;229552618134.786;229541141051.120;229529664547.365;229509653714.778;229489644616.083;229469637252.796;229458164304.138;229455224497.381;229452284725.167;229449344985.509;229446405276.400;229434933464.195;229423462213.461;229411991523.830;229409052244.338;229397582248.173;229386112816.600;229374643949.546;229354646500.304;229334650769.731;229314656759.208;229294664470.127;229283200070.614;229271736219.844;229260272917.578;229248810163.571;229237347957.576;229225886313.924;229205901634.392;229185918713.277;229165937552.373;229154479541.692;229151543406.382;229148607333.353;229145671320.919;229142735367.381;229131278632.290;229119822486.956;229108366931.329;229105431523.551;229093976701.820;229082522444.952;229071068752.880;229051097767.818;229031128514.732;229011160995.157;228991195210.638;228979746044.701;228968297443.070;228956849405.675;228945401932.449;228933955023.319;228922508678.217;228902550565.054;228882594181.038;228871150375.143;228859707140.049;228848264475.492;228845332264.223;228842400082.274;228839467927.619;228836535798.217;228833603692.014;228822162270.133;228810721411.473;228790772938.141;228770826191.516;228750881173.089;228730937884.369;228710996326.879;228691056502.157;228679622135.441;228668188331.208;228656755089.377;228645322409.871;228633890292.606;228622458737.507;228619529237.788;228608098389.634;228596668104.843;228585238383.092;228573809238.589;228562380685.691;228559451993.763;228556523368.491;228545095771.694;228533668781.754;228522242399.062;228510816624.015;228490894430.646;228470974008.705;228451055345.630;228431138428.728;228411223245.183;228399803115.848;228388383551.977;228376964553.528;228365546120.464;228362620189.701;228359694296.042;228348276715.077;228336859698.333;228325443245.509;228314027356.301;228302612030.397;228291197270.100;228279783074.886;228268369444.712;228256956379.544;228245543879.338;228234131944.058;228214233838.589;228194337459.950;228174442809.649;228154549889.220;228143142474.940;228131735625.268;228128812586.218;228125889584.039;228114483589.960;228103078160.748;228091673296.124;228088750748.270;228085828227.577;228074424197.173;228054539934.358;228034657399.176;228023255909.682;228011854984.369;227991976153.606;227980576780.454;227960700664.409;227949302843.159;227929429441.983;227918033172.377;227915112764.244;227903717202.981;227892322222.135;227889402175.708;227886482202.960;227875088179.038;227855221377.272;227843828977.634;227832437158.228;227821045904.701;227809655217.040;227789794141.691;227769934792.560;227750077171.176;227730221279.084;227718835114.311;227715917407.551;227712999738.603;227710082105.583;227707164506.590;227695779481.767;227684395016.986;227664544819.891;227644696351.261;227624849612.658;227613468681.011;227602088314.953;227590708514.470;227579329279.553;227567950610.191;227556572506.374;227545194968.092;227533817997.644;227522441593.196;227511065754.493;227499690481.276;227488315773.285;227476941630.257;227465568050.454;227454195036.117;227442822587.235;227431450703.804;227420079385.812;227408708633.258;227397338446.131;227377513203.776;227357689683.577;227337867887.046;227326501227.342;227315135138.629;227303769620.706;227300857123.476;227297944658.855;227295032224.929;227283667691.821;227272303724.094;227260940321.747;227249577499.227;227238215271.133;227218403944.878;227198594368.396;227178786529.047;227158980414.056;227139176024.967;227127819387.147;227116463323.728;227105107834.550;227102197981.533;227099288163.136;227096378377.470;227085023880.744;227073669950.892;227062316587.938;227050963791.898;227039611562.794;227028259900.650;227016908805.489;226997116803.962;226977326522.122;226965977963.663;226954629973.077;226943282550.148;226931935694.653;226920589406.366;226900805796.729;226889461071.211;226878116912.872;226866773321.740;226855430297.847;226844087841.222;226832745951.899;226821404629.908;226818498393.062;226815592196.601;226804251733.323;226792911831.616;226781572491.209;226761800993.865;226742031220.868;226722263173.777;226702496854.178;226682732263.670;226671398449.186;226660065202.422;226648732523.418;226637400412.222;226634496547.745;226631592725.674;226628688929.811;226625785143.788;226622881365.478;226611550470.535;226591793733.533;226572038716.874;226552285422.087;226532533850.726;226512784004.359;226493035884.572;226481711485.759;226470387654.431;226459064390.633;226456162737.921;226453261126.951;226450359555.942;226447458023.098;226436135913.183;226416394489.358;226396654787.131;226376916808.061;226357180553.713;226337446025.681;226326129428.898;226314813400.260;226303497939.821;226300598308.736;226297698720.479;226294799173.281;226291899665.364;226289000194.932;226269271629.930;226249544783.275;226229819656.505;226210096251.168;226190374568.832;226170654611.080;226150936379.509;226139629149.312;226136731659.208;226133834210.214;226130936800.556;226119630589.121;226108324947.149;226097019874.722;226085715371.924;226074411438.832;226063108075.533;226060211601.942;226048908952.984;226037606867.830;226017900288.066;225998195428.701;225978492291.289;225967193738.065;225955895742.413;225936196272.093;225916498486.720;225896802387.496;225885507793.651;225882613455.108;225879719143.987;225868425373.758;225857132172.408;225854238200.084;225842945716.823;225831653802.716;225828760171.755;225825866582.413;225814575536.565;225803285060.269;225783598807.269;225772309882.025;225752626327.753;225741338953.031;225721658103.830;225701978975.490;225690694148.513;225679409893.720;225659734474.251;225640060782.208;225628779094.801;225617497978.011;225606217431.948;225603326789.571;225600436192.971;225597545640.455;225594655130.316;225583375761.925;225572096965.030;225560818739.753;225541153754.644;225521490486.710;225501828937.513;225490554246.182;225479280128.083;225468006583.121;225456733611.194;225437077741.802;225417423572.303;225406153079.343;225394883115.419;225383613680.210;225372344773.396;225361076394.654;225358188760.032;225355301129.073;225352413499.644;225349525883.915;225346638294.184;225335371054.360;225315725421.237;225296081485.994;225276439250.005;225256798714.662;225237159881.367;225217522751.542;225197887326.623;225186627444.001;225175368123.526;225164109364.998;225161224056.536;225158338774.384;225147080842.798;225135823465.803;225124566643.362;225113310375.434;225102054661.974;225090799502.943;225079544898.301;225076660609.937;225065406729.015;225054153412.368;225042900659.815;225031648471.176;225012029197.018;224992411641.122;224972795805.085;224953181690.523;224941934032.394;224930686944.013;224919440425.508;224908194477.006;224905312471.653;224902430514.135;224899548602.821;224896666750.357;224885422007.693;224865815630.261;224846210983.110;224826608067.975;224807006886.611;224787407440.792;224767809732.310;224756571604.378;224745334063.911;224742454433.067;224739574854.733;224736695313.034;224733815791.935;224730936289.516;224719700020.153;224700108234.670;224680518138.826;224660929733.934;224641343035.607;224621758045.331;224610527173.848;224599296861.404;224588067108.022;224576837913.729;224573960334.265;224571082793.121;224559854450.526;224548626664.746;224537399435.549;224526172762.710;224506597789.463;224495372651.442;224484148072.789;224472924053.536;224461700593.713;224450477693.361;224430909324.075;224411342671.757;224400122325.297;224388902556.978;224386027428.456;224383152349.575;224380277318.740;224369058596.686;224349497433.021;224329937973.587;224310380205.605;224290824130.426;224279609854.117;224268396124.516;224257182941.525;224254309472.367;224243096970.357;224231885029.525;224220673649.706;224209462830.734;224206589958.985;224195379838.744;224184170279.420;224172961281.069;224161752843.755;224142209661.327;224122668181.367;224103128405.368;224083590334.850;224072386370.345;224069515261.900;224066644195.311;224055441089.211;224044238544.739;224041367807.199;224038497104.764;224018964695.306;223999434004.905;223988234305.238;223977035182.300;223957508197.734;223937982918.698;223926786317.243;223915590264.487;223904394760.391;223893199808.093;223890331035.587;223887462288.052;223884593563.661;223865074551.071;223845557227.774;223826041595.138;223806527668.770;223795337544.507;223784147982.794;223772958983.731;223770091709.078;223767224477.398;223756036340.851;223744848764.413;223733661747.932;223714155903.959;223694651767.062;223683467272.609;223672283341.687;223661099974.535;223649917171.533;223638734933.060;223635869430.436;223624687909.931;223613506952.831;223602326559.291;223591146729.460;223571653361.617;223552161690.266;223540984354.914;223529807573.047;223518631344.943;223507455670.882;223496280551.143;223493416858.366;223482242435.234;223471068565.327;223459895248.692;223448722485.374;223429241485.462;223409762185.139;223398591925.962;223387422237.990;223376253121.691;223365084577.532;223353916605.991;223351054783.752;223348193015.850;223337025921.072;223325859367.323;223306389131.291;223286920579.985;223267453715.038;223247988538.097;223236826379.475;223233965993.904;223222804518.066;223211643589.944;223208783495.708;223205923427.693;223194763314.104;223183603763.387;223164145840.170;223144689615.624;223133532579.315;223122376107.940;223111220201.696;223091768619.975;223080614264.884;223069460475.817;223058307253.200;223055449181.421;223052591156.703;223049733177.762;223030287020.757;223010842553.036;222991399762.106;222971958663.710;222952519259.574;222941371895.259;222938515327.775;222935658795.784;222932802297.882;222921655916.918;222910510093.132;222899364826.859;222879931751.331;222860500368.241;222849357599.785;222838215391.537;222827073743.647;222807646980.586;222796506869.516;222785367318.730;222774228328.598;222771373921.465;222768519556.443;222765665232.209;222762810933.269;222743389776.667;222723970300.931;222704552507.683;222685136398.558;222665721975.213;222646309239.319;222626898192.574;222624045612.582;222621193065.839;222618340550.933;222615488066.434;222612635610.899;222601505173.024;222590375301.091;222570969115.176;222551564627.751;222532161840.638;222512760755.684;222501635365.246;222490510547.573;222479386302.992;222476535687.261;222465412170.213;222454289223.668;222443166833.968;222432044987.309;222420923683.902;222418073806.955;222406953177.790;222395833088.917;222384713540.315;222365325307.677;222345938755.195;222326553898.642;222315437804.102;222304322269.166;222293207294.213;222290359012.584;222287510774.677;222284662579.202;222273548605.511;222254170094.666;222234793278.298;222215418158.181;222196044736.109;222176673013.903;222165564473.865;222162717859.316;222159871276.143;222157024722.953;222145917130.662;222134810089.619;222123703600.122;222112597662.477;222101492276.992;222090387442.508;222079283157.413;222068179421.782;222048818778.408;222029459821.327;222010102552.259;221999002271.894;221987902550.546;221976803387.268;221965704779.370;221954606727.207;221951762774.148;221948918859.526;221946074982.014;221934977932.544;221923881469.049;221904533496.441;221885187224.557;221865842641.180;221854749683.771;221843657288.469;221832565455.518;221821474185.163;221802135194.839;221791045466.854;221779956301.299;221768867698.603;221757779659.203;221754938407.082;221752097202.627;221749256044.625;221746414931.848;221727082572.788;221707751906.787;221688422935.661;221669095661.247;221649770085.397;221630446209.985;221619365219.251;221616525864.315;221613686533.493;221610847225.320;221608007938.325;221596928021.660;221585848646.973;221566530291.426;221547213605.944;221527898592.096;221508585251.481;221489273585.703;221486435765.903;221483597975.295;221480760212.486;221477922476.064;221466848948.145;221447540862.586;221428234451.761;221408929717.295;221397859601.714;221386790031.668;221383953434.451;221381116867.291;221378280328.807;221358979901.262;221339681132.970;221320384025.407;221301088594.133;221290023790.907;221278959553.888;221276124298.855;221273289087.652;221262225715.982;221251162908.085;221240100664.424;221229038985.466;221209751614.589;221190465934.885;221179406769.409;221168348172.810;221157290145.434;221146232687.630;221135175797.500;221124119475.963;221113063723.532;221102008540.736;221099175767.246;221096343048.907;221085288742.970;221066014065.944;221046741059.063;221027469723.958;221008200062.276;220997150121.545;220994318665.217;220991487237.198;220988655836.101;220977606841.533;220966558392.913;220955510490.527;220944463134.665;220925200647.789;220905939830.745;220894894926.179;220883850570.145;220872806762.799;220861763504.298;220842508197.486;220823254570.901;220812213789.708;220801173558.037;220798344527.923;220795515519.081;220792686516.014;220781647183.434;220770608373.276;220759570085.591;220740323536.430;220729286744.359;220718250515.784;220707214850.986;220696179750.247;220676938776.405;220657699498.569;220646666927.608;220635634924.796;220624603490.661;220613572625.740;220610746005.365;220607919442.395;220596889468.694;220585860062.450;220574831223.990;220555601029.046;220536372519.605;220517145683.434;220506120305.737;220495095474.113;220484071188.864;220473047450.297;220470222710.430;220459199661.203;220448177159.462;220437155205.323;220426133798.907;220415112940.338;220404092627.583;220393072864.099;220382053650.230;220371034986.317;220360016872.705;220348999309.745;220329788762.253;220318772704.378;220307757198.702;220296742245.413;220285727844.707;220274713996.779;220255509934.344;220244497608.774;220233485823.148;220222474563.699;220211463830.535;220200453623.756;220197632258.870;220186622701.590;220175613669.592;220164605162.802;220153597181.143;220134403459.599;220123396954.555;220112391004.926;220101385625.118;220090380815.695;220079376577.226;220068372911.570;220057369817.844;220046367296.473;220035365347.881;220024363972.505;220013363158.078;220002362890.638;219991363170.506;219980363998.017;219961185549.268;219942008769.940;219931012059.709;219928194194.372;219925376365.214;219922558570.987;219903385156.391;219884213413.059;219865043342.716;219845874947.112;219843058176.495;219840241445.231;219837424752.099;219826433283.559;219815442368.105;219796278313.155;219785288910.278;219774300060.772;219755139604.243;219735980825.879;219733165503.723;219722179288.372;219719364143.123;219708378590.326;219705563580.758;219686409068.263;219667256203.204;219656273169.471;219645290665.694;219626141327.752;219606993643.397;219604179795.264;219601365963.694;219590386114.597;219579406809.856;219568428063.748;219557449890.702;219554636695.588;219543659263.017;219532682401.157;219521706110.431;219502567622.958;219483430818.195;219464295697.975;219445162264.154;219426030518.605;219423219175.729;219420407873.553;219417596610.905;219414785386.585;219411974199.391;219401004059.534;219390034473.083;219379065440.448;219359939492.470;219340815213.157;219321692604.193;219302571667.278;219291607026.481;219280642945.852;219269679425.698;219258716466.326;219247754068.048;219244945115.749;219233983426.749;219223022282.900;219212061684.536;219201101631.990;219190142111.664;219179183123.753;219168224668.460;219157266745.317;219146309353.884;219135352494.176;219124396166.205;219113440369.990;219102485105.552;219083383196.298;219064282943.120;219053330105.433;219042377802.463;219031426034.442;219020474801.599;219009524104.168;218998573942.387;218995767940.883;218984818452.897;218973869500.576;218962921084.005;218951973203.254;218941025858.407;218930079063.468;218919132818.668;218908187124.007;218897241981.904;218886297392.773;218875353357.043;218864409875.141;218853466947.500;218842524574.564;218823445240.703;218804367574.391;218785291577.312;218774352637.303;218763414259.092;218752476442.996;218741539189.333;218730602498.425;218727800023.713;218724997594.711;218722195210.321;218711259529.866;218700324410.282;218689389838.173;218678455799.998;218659390879.131;218640327604.028;218621265976.173;218602205997.064;218583147668.214;218572218864.771;218561290604.456;218550362887.410;218547562663.744;218544762465.984;218541962292.820;218539162142.939;218536362015.015;218525435477.505;218514509477.701;218495458698.802;218476409569.154;218457362104.192;218438316305.614;218419272175.138;218400229714.505;218381188925.473;218378390828.962;218375592781.716;218372794782.687;218369996830.809;218367198925.012;218364401064.211;218361603247.315;218342565903.920;218323530228.237;218304496222.024;218285463887.051;218266433225.112;218247404238.020;218228376927.607;218217465974.880;218206555592.315;218203759943.598;218200964347.997;218198168804.509;218195373312.124;218184464122.169;218173555497.481;218162647424.759;218151739890.560;218140832895.179;218121815015.632;218102798782.799;218083784198.231;218072880569.375;218061977484.049;218051074942.439;218040172944.741;218029271491.150;218018370581.867;218015577386.640;218012784219.551;218001884119.551;217990984562.495;217980085548.731;217969187078.605;217958289138.608;217939287093.836;217920286687.935;217901287922.372;217882290798.625;217863295318.186;217852402577.874;217841510364.369;217838719266.966;217835928181.258;217833137105.817;217830346039.208;217827554979.975;217824763926.652;217821972891.613;217811082015.388;217792092530.544;217773104714.339;217754118568.531;217735134094.889;217716151295.207;217697170171.292;217678190711.120;217659212916.392;217648330301.798;217637448238.264;217626566726.233;217615685766.146;217612897541.397;217610109353.956;217607321202.679;217596441201.341;217585561738.488;217574682814.221;217563804428.634;217552926581.829;217542049273.897;217531172504.934;217520296273.268;217509420583.825;217498545436.956;217487670833.013;217476796772.357;217457836525.057;217438877928.750;217419920971.158;217400965639.873;217382011936.187;217379226574.545;217376441241.192;217373655948.715;217370870696.011;217360002386.441;217349134619.750;217338267396.047;217327400715.456;217316534578.105;217297588220.431;217286723588.477;217275859505.214;217264995971.067;217254132986.465;217243270551.843;217224330619.515;217205392343.577;217186455725.644;217167520767.350;217148587470.346;217137730364.642;217134948242.650;217132166172.066;217129384151.900;217126602167.346;217123820203.439;217112964337.700;217102108980.297;217091254131.016;217080399789.635;217061473966.307;217042549766.743;217031697774.868;217020846310.931;217009995388.953;216999145009.289;216980226202.700;216961309044.738;216942393536.973;216923479680.995;216920700088.024;216917920537.919;216915141029.614;216912361562.041;216901516088.674;216890671158.029;216879826770.270;216868982925.574;216850075496.292;216831169723.908;216820328331.579;216809487487.889;216798647193.275;216787807448.179;216776968253.048;216758068899.614;216747231208.485;216736394071.470;216725557488.609;216714721459.944;216711944678.142;216709167940.152;216706391244.657;216687498123.901;216668606660.793;216649716856.775;216638884673.401;216628053015.641;216617221883.431;216598336508.286;216587506829.847;216576677679.995;216565849058.537;216563074119.689;216560299198.776;216557524294.233;216546696558.901;216527817201.586;216508939471.735;216490063370.501;216471188899.052;216452316058.567;216449542420.672;216446768804.198;216443995207.601;216441221629.316;216430399557.863;216411530214.389;216392662526.893;216373796496.841;216362977841.311;216352159738.622;216349387400.277;216338570002.609;216327753157.976;216316936866.460;216298077527.430;216287262746.746;216276448520.578;216265634849.180;216262863728.277;216260092659.005;216249279849.319;216238467590.787;216219615200.309;216200764468.763;216181915397.671;216163067988.571;216152260124.412;216149490598.455;216146721111.182;216143951647.492;216133144727.290;216122338323.581;216111532449.968;216100727106.451;216081886726.311;216071082835.935;216060279474.617;216049476642.205;216038674338.539;216019839309.297;216001005914.524;215982174155.502;215971375187.742;215968607803.114;215957809495.443;215947011717.826;215944244616.253;215941477540.420;215930680550.001;215919884089.692;215917117296.795;215914350529.610;215895526221.513;215876703552.968;215865909771.639;215855116545.065;215836297445.512;215817480005.387;215798664226.273;215787874441.835;215777085221.886;215766296566.696;215763531900.351;215752743967.754;215741956595.966;215731169785.318;215720383536.145;215709597848.785;215706834041.105;215704070280.528;215701306552.083;215690521827.958;215679737603.918;215660933974.898;215642131950.757;215623331532.528;215604532721.254;215585735517.992;215566939923.810;215556161669.840;215545383958.167;215542621995.632;215539860063.371;215537098159.971;215534336284.001;215523559635.685;215512783521.254;215502007940.763;215491232894.264;215480458381.820;215469684403.486;215458910955.446;215448138037.893;215437365650.795;215426593794.115;215415822467.825;215405051671.887;215386271716.192;215367493395.872;215348716712.329;215329941666.979;215311168274.950;215300402753.875;215289637795.467;215286879102.591;215284120474.057;215281361908.821;215278603405.819;215275844963.983;215273086582.224;215270328259.446;215259565014.349;215240798051.049;215222032746.946;215203269103.693;215184507109.283;215165746751.576;215146988018.317;215136230996.605;215125474493.993;215114718510.327;215111962239.918;215109205978.300;215106449723.833;215095694587.532;215084939963.132;215074185850.467;215063432263.055;215052679214.550;215033930109.946;215023178534.972;215012427499.071;215001677002.318;214990927044.798;214980177626.605;214969428747.826;214950686913.717;214939939520.309;214929192667.036;214918446354.035;214907700581.444;214896955349.399;214894201757.943;214891448208.050;214880703805.308;214869959940.550;214859216613.879;214848473825.400;214829742581.112;214811012991.062;214800272671.640;214789532922.011;214778793742.629;214768055120.283;214757317041.628;214754565401.450;214751813793.812;214741076511.178;214730339755.603;214719603527.078;214708867825.598;214698132651.160;214679414596.474;214660698170.438;214641983374.473;214631251529.318;214620520214.793;214609789430.940;214599059177.807;214596309480.583;214593559816.572;214590810184.409;214588060582.706;214577331395.609;214566602746.911;214555874650.361;214537168988.931;214518464981.392;214499762629.440;214481061921.136;214462362844.432;214451640036.868;214440917762.219;214438170161.986;214435422597.131;214432675066.310;214429927568.178;214427180101.375;214424432664.528;214413711730.552;214395018495.369;214376326882.396;214357636893.016;214338948528.633;214320261790.664;214309546041.407;214298830817.689;214288116133.170;214277402001.645;214274656480.886;214271911005.206;214269165559.750;214258452394.309;214247739763.909;214237027668.652;214226316108.641;214215605083.977;214204894594.765;214186219571.260;214167546171.075;214156838075.790;214146130518.622;214135423499.687;214124717019.093;214114011076.950;214103305673.367;214092600808.452;214089857666.798;214079153459.279;214068449761.171;214057746572.317;214047043906.162;214036341776.297;214025640182.844;214014939125.922;214004238605.655;213993538620.732;213982839171.515;213972140258.038;213961441880.334;213950744038.440;213940046732.387;213929349962.203;213918653727.916;213900003608.831;213881355124.352;213870661311.131;213859968036.112;213849275299.434;213838583101.246;213827891441.695;213817200320.927;213814460685.954;213811721094.657;213808981545.803;213798291379.520;213787601733.355;213776912607.182;213766224000.868;213747587160.741;213728951936.812;213710318330.349;213691686342.629;213673055974.946;213654427228.611;213651689595.432;213648951991.715;213646214416.094;213643476867.192;213640739343.612;213630057460.534;213619376103.598;213608695272.824;213598014968.238;213587335185.646;213576655924.892;213565977185.868;213555298968.463;213536680363.626;213518063387.683;213499448055.599;213480834368.913;213462222329.181;213451549392.069;213440877018.960;213438142156.601;213435407351.175;213432672601.607;213429937906.821;213419266721.664;213408596079.895;213397925968.141;213387256372.901;213376587294.100;213365918731.667;213355250686.791;213344583154.334;213333916134.033;213323249625.611;213312583642.356;213301918197.694;213283321803.739;213264727042.302;213246133914.772;213227542422.541;213208952567.029;213190364349.656;213187632749.835;213184901199.679;213182169698.026;213179438243.702;213176706835.519;213173975472.279;213163317129.122;213152659318.788;213142002027.825;213123419709.464;213104838991.660;213086259875.442;213067682361.849;213057029236.723;213046376633.618;213035724552.310;213025072992.563;213014421954.136;213003771450.344;212993121494.634;212982472087.056;212971823224.913;212961174909.667;212950527141.575;212939879920.897;212929233247.895;212918587109.290;212907941491.669;212897296395.016;212894568558.317;212891840749.357;212881196422.640;212870552610.425;212859909312.448;212849266528.433;212838624258.112;212820068239.325;212801513825.258;212782961030.531;212764409856.378;212745860304.051;212727312374.813;212716676210.244;212706040602.865;212695405552.857;212684771060.403;212682045959.987;212679320910.006;212668687262.497;212658054138.231;212647421537.229;212636789459.504;212626157905.081;212623433531.850;212620709187.188;212617984869.750;212615260578.178;212604630057.928;212586094491.059;212567560525.598;212549028162.569;212530497416.522;212511968288.636;212493440780.101;212482816337.290;212472192449.543;212461569116.964;212450946326.138;212440324063.510;212429702328.894;212419081122.100;212400561780.774;212389942028.412;212379322801.343;212368704099.612;212358085923.269;212347468272.359;212344747440.164;212342026639.811;212339305869.971;212336585129.305;212333864416.457;212315352446.697;212296842078.356;212278333312.418;212259826163.373;212241320645.866;212230708815.421;212220097538.864;212209486816.227;212198876634.035;212188266978.673;212169767770.562;212151270175.890;212132774195.746;212114279831.238;212103674412.574;212093069530.758;212090352015.664;212079747800.305;212069144113.331;212058540954.827;212055823876.072;212053106831.634;212050389820.203;212047672840.458;212037070748.743;212026469172.636;212007984031.579;211989500479.561;211971018530.884;211960420189.731;211949822374.942;211939225086.195;211928628323.158;211918032085.501;211899556342.428;211881082209.944;211862609689.082;211844138780.897;211825669486.445;211815078407.111;211804487865.507;211793897861.453;211783308394.767;211780594778.729;211777881195.032;211767292533.245;211756704402.042;211746116815.011;211735529785.881;211724943301.587;211714357348.930;211703771928.035;211701059399.403;211690474648.612;211679890424.556;211669306726.960;211658723555.554;211648140910.065;211637558790.219;211626977195.747;211608526920.338;211590078256.391;211571631204.975;211553185767.167;211534741944.066;211516299723.322;211497859105.916;211479420092.837;211468846411.428;211458273262.457;211447700645.739;211444991381.760;211442282131.447;211439572906.778;211436863719.858;211434154569.475;211431445454.397;211420874143.854;211410303367.577;211399733125.726;211389163418.474;211370733878.932;211352305945.026;211333879617.751;211315454898.103;211304889400.669;211294324412.196;211283759932.212;211273195960.230;211262632495.760;211252069538.297;211241507100.766;211230945182.788;211212529303.544;211194115027.424;211175702355.375;211157291288.356;211138881840.756;211128325071.968;211117768838.580;211115063707.192;211112358621.897;211109653581.559;211106948585.027;211104243631.143;211101538718.729;211090983884.001;211072580301.960;211054178328.955;211035777965.972;211017379214.003;210998982060.623;210980586493.284;210962192526.150;210951644651.177;210941097306.271;210930550491.147;210920004205.517;210909458449.087;210898913234.972;210896210990.435;210893508788.162;210882964401.825;210872420554.647;210861877233.446;210851334438.313;210832951799.670;210814570758.308;210796191315.060;210785651790.721;210775112794.872;210764574327.175;210754036373.873;210743498934.474;210732962008.476;210722425595.371;210711889708.049;210693519180.106;210682984770.826;210672450902.085;210661917574.130;210651384787.217;210640852541.607;210622488294.246;210611957516.864;210601427268.666;210590897535.874;210580368317.968;210569839614.419;210559311424.690;210548783748.236;210530427471.298;210512072786.584;210493719694.838;210483195265.351;210472671351.127;210462147952.162;210451625068.445;210441102704.243;210430580857.088;210420059526.433;210409538711.738;210399018412.438;210388498641.358;210377979411.442;210367460735.781;210364765278.439;210354247318.994;210343729916.604;210325391577.883;210307054867.803;210288719774.083;210270386284.306;210252054399.310;210233724119.939;210215395447.042;210204885102.580;210202191917.034;210199498777.266;210196805682.160;210186296310.804;210175787470.661;210165279161.313;210154771382.335;210144264119.932;210125943285.501;210107624032.025;210097119071.719;210086614618.481;210068298740.828;210049984440.955;210031671719.444;210021170465.948;210018479510.673;210015788573.388;210005288063.152;209994788056.891;209984288553.929;209973789553.590;209955483322.970;209937178670.774;209918875597.587;209908379795.614;209897884512.676;209887389762.170;209869091033.363;209858597780.328;209848105090.286;209837612963.200;209827121399.034;209816630397.755;209806139959.338;209795650070.405;209792962118.709;209790274195.542;209779785097.844;209761495998.986;209743208491.231;209724922575.323;209706638252.002;209688355522.029;209670074386.165;209659591288.085;209649108719.639;209638626680.477;209635940729.522;209633254812.836;209630568929.263;209627883077.636;209609607220.458;209591332953.760;209573060278.287;209554789194.787;209536519704.024;209526043283.130;209515567389.966;209505092024.184;209494617185.436;209484142873.372;209465879560.048;209455406693.408;209444934352.743;209434462524.861;209431779170.796;209421307942.679;209410837198.115;209400366936.302;209389897156.431;209379427857.677;209361173419.289;209342920560.503;209324669295.265;209306419637.664;209288171588.620;209277707392.059;209275025878.728;209272344416.597;209269663004.709;209266981642.105;209264300327.810;209246056060.842;209227813396.213;209209572334.772;209191332877.373;209173095024.882;209162636742.782;209152179003.105;209141721805.611;209131265150.053;209120809036.180;209110353450.434;209092122749.911;209073893634.839;209055666105.873;209037440163.680;209026988727.825;209016537815.103;209013859855.119;209011181922.286;209000731786.298;208990282167.025;208979833051.211;208961613661.646;208951165939.645;208940718720.029;208930272002.089;208919825785.113;208909380068.371;208898934851.130;208880722365.987;208862511463.075;208844302156.283;208826094446.349;208807888334.017;208797448203.777;208794772892.440;208792097624.536;208789422399.002;208786747214.770;208768544584.676;208750343546.615;208732144101.267;208721707816.072;208711272060.705;208700836834.696;208682641737.433;208664448236.578;208646256332.859;208635824400.337;208625393003.292;208614962141.285;208604531813.884;208586345201.886;208568160191.738;208557732251.318;208547304844.949;208544632956.131;208541961116.800;208531534542.028;208521108498.355;208510682972.010;208492504618.764;208474327822.993;208456152584.988;208437978905.048;208427557412.559;208417136417.061;208406715917.680;208396295913.530;208378127326.141;208359960298.891;208341794832.088;208339125315.747;208336455807.294;208333786305.267;208331116808.190;208312953730.700;208294792231.182;208276632323.368;208258474021.138;208240317325.263;208229905518.151;208219494259.774;208209083549.751;208198673387.689;208188263773.188;208177854705.851;208167446185.262;208149297631.960;208131150666.068;208113005274.911;208094861458.930;208076719218.561;208058578554.248;208040439466.443;208037773937.710;208035108437.892;208032442965.751;208029777520.025;208019376361.468;208008975709.053;207998575561.912;207988175919.160;207977776779.904;207959644632.102;207941514054.633;207923385047.867;207905257612.174;207887131747.929;207869007455.512;207850884735.300;207832763587.677;207814644013.032;207804253464.678;207793863448.757;207783473977.850;207780811641.723;207778149373.170;207775487158.112;207765098700.155;207754710749.760;207744323306.003;207733936367.959;207715825487.171;207697716177.838;207679608440.321;207661502274.972;207643397682.153;207625294662.228;207607193215.561;207589093342.519;207570995043.478;207560616712.804;207557957193.507;207555297706.371;207544920156.536;207534543119.782;207524166595.218;207513790581.942;207503415079.039;207493040098.785;207482665653.578;207464576640.700;207446489228.202;207436117175.706;207425745664.481;207407661690.393;207389579287.151;207371498441.880;207353419154.763;207335341425.985;207317265255.732;207306899604.999;207296534461.813;207286169825.160;207275805694.014;207265442067.341;207255078957.272;207244716362.883;207234354283.234;207223992717.376;207213631664.349;207203271136.364;207192911132.363;207182551657.221;207164488765.162;207146427457.834;207128367735.676;207110309599.125;207092253048.624;207074198084.623;207056144707.569;207038092917.918;207027741290.556;207017390200.674;207007039634.357;207004387281.416;207001734938.129;206991385092.617;206981035747.559;206970686914.987;206960338593.886;206942295101.391;206924253181.134;206906212833.400;206888174058.477;206870136856.649;206852101228.212;206834067173.454;206816034692.673;206805694106.982;206795354045.189;206785014506.372;206774675489.589;206764336993.887;206753999018.296;206743661561.836;206733324623.506;206715300960.971;206697278875.951;206679258368.764;206668924668.283;206658591488.437;206648258816.227;206630242570.587;206619911291.184;206609580531.550;206599250290.668;206588920567.496;206578591360.991;206568262670.087;206550253393.712;206532245690.119;206514239559.566;206496235002.302;206478232018.590;206460230608.685;206442230772.845;206431908933.999;206421587620.840;206411266832.381;206400946567.625;206390626825.563;206380307605.172;206369988905.415;206359670725.247;206341679716.579;206323690285.153;206305702431.249;206287716155.149;206269731457.137;206251748337.502;206233766783.411;206215786781.896;206205476320.021;206195166349.553;206184856869.116;206174547877.322;206164239372.762;206153931354.014;206143623819.638;206133316781.290;206115345324.543;206097375446.357;206079407147.003;206061440426.760;206043475285.909;206025511724.738;206015210663.460;206004910127.094;205994610115.858;205984310632.157;205974011675.563;205963713245.049;205953415339.579;205943117958.102;205932821099.563;205922524762.891;205912228947.010;205901933637.728;205883982495.707;205866032889.509;205848084818.982;205830138283.974;205812193284.332;205794249833.000;205776307943.050;205758367614.600;205740428847.777;205730141966.884;205719855603.185;205709569755.589;205699284422.995;205688999604.299;205678715298.384;205668431504.124;205658148220.383;205647865446.024;205629936303.225;205612008724.105;205594082708.821;205576158257.533;205558235370.408;205540314047.607;205522394302.376;205504476148.096;205486559585.218;205476285500.158;205466011951.484;205455738925.284;205445466407.508;205435194397.027;205424922892.710;205414651893.413;205404381397.984;205394111405.261;205383841914.072;205365935862.390;205348031365.526;205330128423.566;205312227036.600;205294327204.719;205276428928.014;205258532206.589;205248269442.618;205238007182.679;205235377366.848;205225115779.439;205214854723.195;205204594197.260;205194334200.769;205184074732.846;205173815792.610;205163557379.169;205153299491.622;205143042116.008;205125157131.707;205107273685.301;205089391776.718;205071511405.893;205053632572.756;205035755277.240;205017879519.279;205000005298.808;204982132615.764;204971883612.322;204961635132.147;204951387174.323;204941139737.929;204930892822.035;204920646425.706;204910400547.998;204900155187.963;204889910344.640;204879666017.070;204869422204.276;204851560977.611;204833701317.597;204815843224.494;204797986698.552;204780131740.031;204762278349.194;204744426526.309;204726576271.646;204708727572.454;204690880415.856;204680646084.116;204670412244.097;204660178907.628;204649946086.674;204639713780.305;204629481987.582;204619250707.557;204609019939.281;204598789681.793;204588559934.126;204578330695.307;204568101964.353;204550267013.353;204532433620.100;204522207186.999;204511981264.230;204501755851.903;204483926685.951;204466099070.440;204448273018.491;204430448543.357;204420227258.708;204410006514.591;204399786310.392;204389566632.485;204379347467.104;204369128800.345;204358910631.163;204348692958.507;204338475781.310;204328259098.499;204318042908.988;204307827211.684;204290014959.696;204272204248.385;204254395077.773;204236587447.895;204218781358.784;204200976810.469;204183173802.987;204165372336.377;204147572410.675;204129774025.925;204119567632.930;204109361773.217;204099156446.114;204088951650.946;204078747387.027;204068543653.667;204058340450.168;204048137775.826;204037935629.930;204027734011.761;204017532907.606;204007332303.621;203997132185.813;203986932553.029;203976733404.105;203958950000.198;203941168141.548;203923387828.259;203905609060.439;203887831838.192;203870056161.635;203852282030.877;203834509459.004;203816738446.268;203798968992.929;203781201099.244;203778589984.887;203775978918.274;203773367898.573;203763179464.394;203752991547.940;203742804148.438;203732617265.113;203722430897.179;203712245043.849;203702059704.332;203691874877.823;203681690563.512;203671506747.630;203661323429.224;203651140607.334;203633385650.737;203615632236.940;203597880366.053;203580130038.182;203562381253.438;203544634011.934;203526888313.788;203509144159.118;203491401548.044;203473660480.694;203463486994.914;203453314022.465;203443141562.593;203432969614.536;203422798177.526;203412627250.796;203402456833.567;203392286925.058;203382117524.481;203371948631.045;203361780243.952;203351612375.335;203341445037.457;203331278216.838;203321111899.853;203310946072.746;203293220755.092;203275496972.172;203257774724.034;203240054010.733;203222334832.327;203204617188.869;203186901080.423;203169186507.049;203151473481.735;203133762004.682;203123605473.558;203113449462.466;203103293970.794;203093138997.931;203082984543.251;203080382390.471;203077780288.744;203075178237.402;203065024739.159;203054871760.939;203044719302.142;203034567362.153;203024415940.363;203014265023.231;203004114597.077;202993964648.074;202983815175.179;202966118359.992;202948423063.797;202930729299.418;202913037079.821;202895346405.193;202877657275.723;202859969691.609;202842283653.047;202824599160.239;202806916213.393;202796776060.644;202786636427.589;202776497313.631;202766358718.174;202756220640.606;202746083080.317;202735946036.687;202725809509.096;202715673496.910;202705537999.492;202695403016.199;202685268546.385;202675134589.391;202665001144.559;202654868198.323;202644735736.975;202634603746.674;202624472213.435;202606806693.035;202589142697.003;202571480225.346;202553819278.072;202536159855.192;202518501956.716;202500845582.657;202483190733.030;202465537407.851;202447885607.140;202437763222.427;202427641336.792;202417519949.475;202407399072.591;202404805379.926;202402211732.386;202399618129.296;202389498194.703;202379378772.473;202369259862.003;202359141462.690;202349023573.928;202338906195.107;202328789325.610;202318672964.822;202308557112.122;202298441766.881;202280804596.061;202263168972.217;202245534895.624;202227902366.559;202210271385.301;202192641952.138;202175014067.358;202157387731.257;202147280032.497;202137172857.460;202127066205.711;202116960076.827;202106854470.373;202096749373.057;202086644784.301;202076540703.534;202066437130.172;202056334050.783;202046231464.649;202036129371.049;202026027769.256;202015926658.541;201998314210.324;201980703289.027;201963093894.743;201945486027.566;201927879687.597;201910274874.929;201900179491.862;201890084610.693;201879990230.825;201869896351.663;201859802972.593;201849710093.016;201839617712.319;201829525829.889;201819434445.109;201809343557.356;201799253166.004;201789163270.427;201771570474.212;201753979207.685;201736389470.993;201726302697.858;201716216434.249;201713631658.632;201703546082.252;201693461041.622;201683376536.552;201673292566.856;201663209132.341;201653126232.827;201643043868.121;201632962038.040;201622880729.569;201612799929.561;201595222787.256;201577647160.382;201560073048.955;201542500452.994;201532423597.303;201522347231.527;201512271355.036;201502195967.199;201492121067.384;201482046654.949;201471972729.255;201461899289.655;201444335109.509;201426772445.562;201409211297.854;201399140936.021;201389071078.072;201379001736.370;201368932910.600;201358864600.450;201348796805.599;201338729525.728;201328662760.528;201311110238.918;201293559262.208;201276009830.749;201265946247.320;201255883172.225;201245820592.280;201235758506.962;201225696915.745;201215635818.101;201205575213.495;201188033406.518;201170493121.064;201160434748.583;201150376872.422;201140319492.081;201130262619.848;201112727374.296;201095193677.267;201085139121.859;201082562585.013;201079986099.151;201077409663.907;201067356070.449;201057302993.217;201047250431.929;201029722580.075;201012196272.688;200994671510.155;200984622134.474;200974573255.788;200964524860.824;200954476948.961;200944429519.570;200934382572.020;200924336105.672;200914290119.881;200896773750.285;200879258888.435;200861745534.347;200851702601.030;200841660153.195;200831618190.246;200821576711.581;200811535729.372;200801495255.919;200791455303.647;200781415872.339;200771376961.777;200753873024.994;200736370637.125;200718869798.627;200701370509.966;200691335719.884;200681301460.229;200671267718.109;200661234480.507;200651201734.269;200641169478.872;200631137713.796;200621106438.505;200611075652.470;200601045355.148;200591015545.999;200573527382.273;200556040733.628;200538555600.191;200521071982.094;200511046124.203;200501020762.139;200490995895.445;200480971523.660;200470947659.080;200468378970.350;200465810341.446;200463241772.144;200453218889.173;200443196530.009;200433174681.748;200415700360.787;200398227564.673;200380756293.651;200363286547.975;200345818327.903;200335801364.519;200325784910.390;200315768965.251;200305753528.832;200295738600.871;200285724168.350;200275710230.872;200265696788.036;200255683839.439;200245671384.682;200235659423.360;200225647955.067;200215636966.657;200205626457.584;200195616427.312;200185606875.293;200175597800.978;200158145807.257;200140695330.534;200123246383.741;200105798979.943;200088353106.884;200078348918.213;200068345243.975;200058342084.026;200048339438.226;200038337306.431;200028335688.510;200018334584.319;200008333980.998;199998333865.547;199988334237.566;199978335096.659;199968336442.422;199958338274.454;199940905274.913;199923473785.441;199906043806.199;199888615337.343;199878621108.587;199868627374.649;199858634135.230;199848641390.040;199838649138.782;199828657381.166;199818666116.895;199808675358.391;199806115146.532;199803554994.332;199800994901.728;199791005124.238;199781015869.916;199771027138.796;199761038918.208;199743623215.132;199726209021.043;199708796336.114;199691385160.516;199681400887.127;199671417108.087;199661433823.158;199651451032.100;199641468734.681;199631486930.661;199621505619.804;199611524801.877;199601544476.644;199591564643.875;199581585303.334;199571606454.791;199561628098.013;199551650232.770;199534252665.709;199516856612.108;199499462072.214;199482069058.968;199472095184.341;199462121823.791;199452148964.645;199442176606.797;199432204750.152;199422233394.607;199412262540.067;199402292186.438;199392322333.627;199382352981.543;199372384130.099;199362415779.209;199352447928.792;199342480578.765;199332513729.049;199322547379.573;199312581530.258;199302616181.035;199292651331.835;199282686983.351;199272723136.422;199255349967.462;199237978315.779;199228016714.744;199218055606.109;199208094977.057;199198134814.661;199188175131.203;199178215939.096;199168257250.889;199158299066.595;199148341386.229;199138384209.804;199128427537.343;199118471368.868;199108515704.410;199098560543.996;199088605887.665;199078651735.451;199068698087.396;199051342706.925;199041390442.143;199024037462.036;199014086582.033;199004136211.890;198994186351.731;198984237001.680;198974288149.202;198964339781.635;198954391886.183;198944444462.595;198934497510.611;198924551029.975;198914605020.434;198904659481.733;198894714413.619;198884769828.492;198874825738.470;198857487129.761;198847544429.109;198837602239.638;198827660561.539;198817719394.997;198807778740.220;198797838597.407;198787898966.769;198777959848.519;198768021242.880;198750692094.107;198733364474.411;198723428155.062;198713492342.002;198703557022.760;198693622184.731;198683687827.833;198673753951.983;198671208383.370;198661275097.339;198651342289.445;198641409959.594;198631478107.687;198621546733.635;198611615837.342;198601685431.360;198591755528.366;198581826141.179;198571897270.120;198554585151.990;198537274566.905;198527347992.553;198517421941.061;198507496412.836;198497571395.660;198487646877.186;198485103749.831;198475179832.162;198465256398.128;198455333447.750;198445410981.047;198435488998.046;198418188808.504;198400890117.002;198390970331.140;198381051034.503;198371132227.203;198361213909.337;198351296081.019;198341378742.358;198331461893.478;198321545534.500;198311629665.547;198309088646.335;198299173399.084;198289258664.962;198279344456.969;198269430775.630;198259517621.474;198249604982.420;198239692846.268;198229781200.687;198219870033.226;198209959343.930;198200049132.838;198190139400.001;198180230145.472;198162952171.375;198145675690.049;198128400701.730;198118494482.349;198108588749.715;198106050313.129;198096145183.470;198086240538.377;198076336377.999;198066432702.492;198056529512.011;198046626806.721;198036724586.791;198026822852.395;198016921616.313;198007020891.457;197997120690.885;197987221015.184;197977321864.966;197967423240.842;197957525143.432;197954988765.515;197945091352.736;197935194465.929;197925298105.730;197915402260.183;197905506917.209;197895612064.613;197885717690.065;197875823793.723;197865930375.739;197856037436.274;197846144975.494;197836252993.569;197826361490.675;197816470466.994;197806579922.712;197796689858.019;197786800273.115;197776911168.199;197767022543.482;197764488507.688;197754600473.651;197744712930.379;197734825890.793;197724939367.941;197715053362.433;197697815973.755;197680580116.429;197663345778.587;197653462929.483;197643580570.821;197633698702.951;197623817326.209;197613936440.950;197604056047.531;197594176146.310;197591644418.140;197581765132.386;197571886337.273;197562008033.152;197552130220.383;197542252899.324;197532376070.344;197522499733.815;197512623890.114;197502748539.624;197492873682.730;197482999319.824;197473125451.301;197463252077.550;197453379198.975;197443506815.990;197433634929.013;197423763538.467;197413892644.785;197411363225.039;197401492956.698;197391623183.881;197381753907.007;197371885126.508;197362016842.824;197352149056.394;197342281767.665;197332414977.092;197322548685.137;197312682892.262;197302817598.939;197292952805.645;197283088512.867;197273224721.092;197263361430.821;197260834009.405;197250971316.423;197241109085.731;197231247317.424;197221386011.598;197211525168.356;197201664800.347;197191804920.357;197181945541.302;197172086663.693;197162228288.039;197145039068.455;197127851361.839;197110665168.805;197100809921.805;197090955187.499;197081100966.529;197071247259.542;197061394054.644;197051541339.818;197049016605.509;197039164477.815;197029312825.961;197019461650.204;197009610950.809;196999760728.052;196989910982.197;196980061713.528;196970212922.325;196960364608.872;196950516773.459;196940669416.380;196930822550.465;196920976188.681;196911130344.126;196908607298.585;196898762121.870;196888917461.875;196879073319.312;196869229694.893;196859386589.342;196849544003.392;196839701925.248;196829860342.998;196820019244.603;196810178617.904;196800338463.144;196790498780.572;196780659570.443;196770820833.009;196760982568.533;196751144777.279;196741307459.515;196731470615.511;196721634245.546;196719113584.844;196709277795.395;196699442478.176;196689607633.457;196679773261.500;196669939375.097;196660105987.173;196650273110.781;196640440746.603;196630608895.331;196620777557.661;196610946734.304;196593805578.959;196576665946.559;196559527837.914;196557009398.480;196547181897.443;196537354895.454;196527528380.561;196517702340.691;196507876776.156;196498051687.276;196488227074.371;196478402937.772;196468579277.809;196458756094.819;196448933389.145;196439111161.129;196429289411.126;196419468139.491;196409647346.582;196399827045.268;196390007248.545;196380187969.547;196370369209.057;196360550967.855;196358035053.730;196355519200.093;196345701783.486;196335884871.576;196326068452.509;196316252514.305;196306437057.364;196296622082.091;196286807588.900;196276993578.206;196267180050.431;196257367006.001;196247554445.350;196237742368.916;196227930777.144;196218119670.487;196208309049.395;196198498914.334;196188689265.771;196178880104.180;196169071430.040;196159263231.345;196149455495.966;196139648224.133;196129841428.563;196120035122.111;196110229317.769;196100424016.181;196090619218.000;196080814923.892;196071011134.521;196053917081.641;196036824539.228;196019733508.060;196009932853.886;196000132715.953;195990333095.063;195980533992.030;195970735407.679;195960937342.841;195951139798.361;195941342762.612;195931546223.849;195921750170.204;195911954602.166;195902159520.229;195892364924.889;195882570816.656;195872777196.043;195862984063.568;195853191419.756;195850682205.452;195848173019.218;195845663861.425;195835872069.404;195826080760.753;195816289935.952;195806499595.490;195796709739.858;195786920357.086;195777131435.086;195767342961.636;195757554936.862;195747767360.890;195737980233.844;195728193555.862;195718407327.070;195708621560.063;195698836267.565;195689051462.427;195679267157.648;195669483366.349;195659700089.346;195649917327.461;195640135081.526;195630353352.381;195620572140.880;195610791447.886;195601011274.268;195591231620.909;195581452476.244;195571673828.592;195561895666.143;195552117976.979;195542340761.497;195532564020.113;195522787753.241;195505741578.586;195488696880.942;195471653660.813;195461880402.517;195452107628.890;195442335340.473;195432563537.799;195422792221.416;195413021391.878;195403251049.740;195393481195.570;195383711829.940;195373942953.431;195364174566.629;195354406670.126;195351903658.103;195349400677.212;195346897727.897;195337130693.231;195327364143.941;195317598068.143;195307832466.254;195298067338.707;195288302685.928;195278538520.797;195268774843.891;195259011655.806;195249248957.133;195239486748.474;195229725030.448;195219963803.665;195210203068.753;195200442826.345;195190683077.081;195180923821.612;195171165060.589;195161406794.680;195151649024.554;195141891750.889;195132134974.375;195122378695.710;195112622903.165;195102867584.899;195093112728.942;195083358335.632;195073604405.305;195063850938.309;195054097934.990;195044345395.700;195034593333.208;195024841760.422;195015090690.380;195005340123.847;194995590061.596;194978589725.831;194961590894.711;194944593569.116;194934846635.207;194925100205.154;194915354267.322;194905608809.963;194895863833.616;194886119338.826;194876375326.149;194866631796.148;194856888749.384;194847146186.431;194837404107.876;194834907682.248;194832411282.777;194829914909.887;194820173672.326;194810432914.189;194800692636.013;194790952850.742;194781213571.457;194771474811.375;194761736559.046;194751998802.909;194742261531.276;194732524732.345;194722788406.592;194713052554.500;194703317176.551;194693582273.243;194683847845.073;194674113892.549;194664380416.183;194654647416.491;194644914893.995;194635182849.229;194625451282.728;194615720195.037;194605989586.703;194596259458.284;194586529810.340;194576800643.445;194567071958.174;194557343755.107;194547616034.838;194537888810.345;194528162094.744;194518435901.287;194508710230.983;194498985072.469;194489260414.258;194479536244.747;194469812552.214;194460089337.203;194450366600.260;194433413834.275;194416462543.580;194399512728.822;194389793009.993;194380073780.039;194370355039.635;194367864590.081;194365374173.575;194362883790.641;194353165915.441;194343448525.313;194333731620.892;194324015202.830;194314299271.762;194304583828.344;194294868860.864;194285154369.868;194275440355.912;194265726819.549;194256013761.347;194246301181.877;194236589081.719;194226877461.458;194217166321.684;194207455675.362;194197745523.222;194188035866.013;194178326704.488;194168618039.403;194158909859.161;194149202152.048;194139494918.577;194129788159.285;194120081874.710;194110376065.399;194100670731.904;194090965874.785;194081261494.607;194071557591.941;194061854167.370;194052151233.828;194042448792.037;194032746842.728;194023045386.641;194013344424.520;194003643957.121;193993943985.210;193984244509.559;193974545530.937;193964847050.128;193955149055.573;193945451547.952;193935754527.954;193926057996.269;193916361953.605;193899455751.553;193882551033.356;193865647799.821;193855954817.599;193846262335.968;193836570355.763;193834086828.908;193831603335.506;193829119863.761;193819428738.703;193809738086.107;193800047906.517;193790358200.489;193780668968.574;193770980211.338;193761291929.354;193751604123.196;193741916793.441;193732229940.672;193722543577.812;193712857705.599;193703172324.773;193693487436.083;193683803040.289;193674119138.156;193664435730.463;193654752817.991;193645070401.534;193635388481.894;193625707047.546;193616026086.851;193606345600.378;193596665588.701;193586986052.402;193577306992.078;193567628408.318;193557950301.729;193548272672.924;193538595522.518;193528918838.807;193519242622.285;193509566885.779;193499891642.248;193490216892.475;193480542637.248;193470868877.361;193461195613.624;193451522846.850;193441850577.863;193439372079.492;193436893625.257;193434415203.521;193424743802.402;193407880499.771;193391018657.404;193374158275.924;193364489848.828;193354821900.928;193345154432.867;193335487445.299;193325820938.884;193316154901.970;193306489322.788;193296824201.762;193287159551.624;193277495385.235;193267831715.604;193258168543.555;193248505869.935;193238843695.597;193229182021.397;193219520848.212;193209860176.918;193200199996.100;193190540294.218;193180881071.928;193171222329.882;193161564068.754;193151906289.213;193142248991.940;193132592177.623;193122935846.960;193113279988.355;193103624590.084;193093969652.610;193084315176.405;193074661161.934;193065007621.970;193055354569.414;193045702017.301;193036049966.520;193026398417.960;193016747372.528;193007096831.136;192997446794.708;192987797251.885;192978148191.187;192968499601.015;192958851481.948;192949203834.562;192939556659.450;192929909957.207;192920263728.434;192910617973.741;192900972693.740;192891327889.059;192881683560.323;192872039708.172;192869568418.598;192867097151.411;192864625907.082;192847811346.447;192830998270.765;192814186680.964;192804546234.009;192794906269.496;192785266775.851;192775627753.664;192765989203.529;192756351126.047;192746713521.828;192737076391.490;192727439735.649;192717803554.939;192708167849.994;192698532621.459;192688897869.986;192679263596.229;192669629800.857;192659996484.541;192650363647.958;192640731291.800;192631099429.026;192621468072.730;192611837236.143;192602206920.370;192592577126.536;192582947843.497;192573319060.002;192563690764.678;192554062946.038;192544435592.473;192534808704.526;192532341827.611;192529874964.923;192520248753.402;192510623004.485;192500997718.684;192491372896.522;192481748538.523;192472124645.226;192462501217.162;192452878254.883;192443255758.936;192433633729.879;192424012168.276;192414391074.695;192404770461.964;192395150343.043;192378376645.561;192361604453.219;192351986605.733;192342369285.862;192332752482.555;192323136184.640;192313520380.837;192303905059.745;192294290209.845;192284675831.752;192275061926.086;192265448493.477;192255835534.557;192246223049.969;192236611040.363;192226999506.393;192224536546.134;192222073611.120;192212462785.510;192202852433.009;192193242554.258;192183633149.899;192166878000.064;192157269906.346;192147662291.597;192138055156.517;192128448501.813;192118842328.206;192109236636.417;192099631427.179;192090026688.994;192080422422.471;192070818628.229;192061215306.891;192051612459.090;192042010085.473;192032408186.694;192022806763.418;192006065576.613;191996465465.945;191986865835.404;191977266685.734;191967668017.709;191958069832.116;191948472129.748;191938874911.407;191929278177.904;191926818957.196;191924359771.757;191914763776.276;191905168263.752;191895573234.984;191885978690.783;191876384631.966;191866791059.362;191857197973.796;191847605376.104;191830879541.387;191821288269.025;191811697465.136;191802107130.365;191792517265.381;191782927870.846;191773338947.432;191763750495.822;191754162516.705;191744575010.780;191734987978.747;191725401421.318;191722944784.655;191720488173.820;191710902323.871;191701316946.240;191691732041.626;191682147610.731;191665436094.002;191655853008.379;191646270438.218;191636688384.665;191627106836.670;191617525783.066;191607945212.577;191598365113.802;191588785487.434;191579206334.174;191569627654.729;191560049449.816;191550471720.159;191540894466.487;191531317689.539;191521741390.063;191512165568.813;191495468974.727;191485894470.862;191476320450.122;191466746913.329;191464293677.225;191461840452.032;191452267590.820;191442695198.836;191433123288.939;191423551861.935;191413980918.639;191404410459.874;191394840486.469;191385270999.269;191375701999.124;191366133486.892;191356565463.442;191346997917.458;191337430837.504;191320749469.988;191311183682.328;191301618365.226;191292053519.370;191282489145.457;191272925256.368;191263361852.932;191260911199.449;191258460584.094;191248897923.771;191239335747.340;191229774055.628;191220212849.478;191210652129.731;191201091897.247;191191532152.894;191174863568.063;191165305160.628;191155747234.372;191146189777.919;191136632779.770;191127076240.493;191117520160.654;191107964540.830;191098409381.605;191088854683.568;191079300447.314;191069746673.445;191060193362.567;191050640527.463;191041088181.047;191031536336.366;191021985006.604;191019537481.005;191017090011.443;191007539478.772;190990886935.470;190981337716.329;190971788978.852;190962240723.851;190952692952.146;190943145664.567;190933598861.951;190924052545.146;190914506715.008;190904961372.404;190895416506.039;190885872104.507;190876328168.444;190866784698.494;190857241695.309;190847699159.547;190831060547.851;190821519306.322;190811978536.812;190809533706.334;190807088876.747;190797548749.052;190788009078.281;190778469877.136;190768931158.458;190759392935.216;190749855208.372;190740317978.894;190730781247.767;190721245003.824;190711709235.780;190702173944.395;190692639130.436;190683104794.671;190673570937.885;190656947491.565;190647414952.168;190637882884.564;190628351289.472;190618820167.610;190609289519.712;190599759334.364;190590229612.189;190580700353.811;190571171559.858;190568729755.793;190566287981.577;190556759904.934;190547232314.779;190537705211.984;190528178597.428;190511567796.502;190502042493.470;190492517671.546;190482993331.542;190473469462.143;190463946064.054;190454423137.985;190444900684.661;190435378704.806;190425857187.018;190416336119.777;190406815503.582;190397295351.058;190387775674.962;190378256488.185;190368737791.632;190366298582.480;190363859418.254;190354341482.302;190344824035.109;190328229232.277;190318713121.741;190309197493.122;190299682335.114;190290167636.299;190280653397.266;190271139618.605;190261626300.922;190252113444.821;190242601050.919;190233089119.837;190223577652.199;190214066660.754;190204556158.383;190195046158.104;190178464385.262;190168955736.329;190159447570.831;190149939877.492;190140432644.922;190130925873.728;190128489719.075;190126053580.200;190116547482.152;190107041842.808;190097536662.752;190088031942.579;190078527682.887;190069023884.277;190059520547.366;190050017684.873;190040515309.653;190031013434.694;190021512061.020;190012011189.661;189995445351.604;189985945843.840;189976446844.123;189966948341.446;189957450324.691;189947952782.621;189938455703.886;189928959077.018;189919462902.531;189917029471.313;189914596047.317;189905100519.248;189895605440.525;189886110811.640;189876616633.089;189860062469.775;189850569561.071;189841077131.095;189831585192.739;189822093759.027;189812602831.030;189803112409.827;189793622496.516;189784133092.199;189774644197.997;189765155815.034;189755667932.355;189746180538.893;189736693623.462;189727207174.762;189717721181.373;189708235643.853;189698750562.765;189696320021.365;189693889491.739;189684405070.322;189667867815.039;189651331983.175;189641849637.087;189632367751.372;189622886326.637;189613405363.503;189603924862.591;189594444836.608;189584965298.392;189575486260.913;189566007737.280;189556529728.661;189547052224.161;189537575212.759;189528098683.331;189518622624.635;189509147025.306;189499671885.952;189490197207.175;189487769318.475;189485341446.600;189475867441.003;189466393893.334;189456920804.181;189447448174.135;189437976003.798;189428504293.775;189419033044.676;189409562269.189;189400091980.127;189390622190.437;189381152901.142;189371684113.276;189362215827.881;189352748033.936;189343280720.314;189333813875.764;189324347488.926;189314881560.384;189305416090.732;189295951080.569;189286486530.498;189277022441.136;189267558813.098;189258095647.010;189255670653.105;189253245701.609;189243783286.616;189234321367.921;189224859934.457;189215398975.029;189198902630.714;189182407719.051;189165914240.775;189156456206.759;189146998645.832;189137541558.744;189128084946.251;189118628809.121;189109173148.129;189099717964.054;189090263257.693;189080809017.780;189071355232.951;189061901903.755;189059479443.678;189057057020.693;189047604423.919;189038152316.836;189028700700.370;189019249575.454;189009798943.032;189000348804.062;188990899159.504;188981450010.337;188972001345.498;188962553153.804;188953105423.964;188943658156.614;188934211352.396;188924765011.957;188915319135.948;188905873725.037;188896428779.892;188886984301.187;188877540289.607;188868096745.843;188858653670.593;188849211052.522;188839768880.174;188830327154.021;188827907677.795;188825488228.919;188816047207.075;188806606665.045;188797166591.612;188787726987.488;188778287853.387;188768849190.029;188759410998.258;188749973279.046;188740536033.369;188724081071.146;188707627551.169;188691175474.532;188681741188.866;188672307365.330;188662873992.764;188653441071.923;188644008603.570;188634576600.492;188632159587.710;188622728200.812;188613297303.787;188603866897.797;188601450353.968;188592020580.384;188582591298.846;188573162498.514;188563734168.436;188554306297.547;188544878874.666;188535451900.519;188526025375.835;188516599301.356;188507173677.827;188497748518.013;188488323834.808;188478899641.240;188469475938.473;188460052727.678;188450630010.040;188441207786.755;188431786059.033;188429371799.260;188419950701.460;188410530076.230;188408116093.402;188398696027.276;188389276409.750;188379857241.554;188370438523.430;188361020256.123;188351602440.385;188342185076.986;188332768178.690;188323351758.398;188313935829.149;188304520392.115;188295105448.489;188285690987.461;188276276998.117;188266863469.428;188257450390.250;188248037761.320;188231625706.047;188215215066.985;188198805844.996;188196394397.313;188193982981.286;188184573479.196;188175164459.888;188165755924.461;188156347874.011;188146940309.655;188137533232.517;188128126631.742;188118720496.359;188109314815.281;188099909589.293;188090504831.178;188081100541.871;188071696722.320;188062293373.478;188052890496.315;188043488091.802;188034086160.929;188024684704.691;188015283724.097;188005883220.163;187996483193.915;187987083646.388;187984675011.788;187982266404.095;187972867568.312;187963469198.153;187954071294.539;187944673858.387;187935276890.632;187925880392.216;187916484364.090;187907088807.208;187897693722.529;187888299099.056;187878904937.658;187869511239.210;187860118004.597;187850725234.718;187841332930.475;187831941092.784;187822549722.567;187813158820.758;187810752354.582;187801362052.637;187784988964.535;187768617305.704;187759229119.282;187749841397.699;187740454141.864;187731067352.693;187721681031.106;187712295178.046;187702909794.458;187693524893.251;187684140475.510;187674756530.362;187665373058.804;187655990061.831;187646607540.451;187637225495.687;187627843928.570;187618462828.173;187609082183.466;187599701995.260;187590322264.372;187580943003.585;187571564225.811;187562185932.151;187559782757.936;187557379616.990;187548002041.345;187538624924.929;187529248268.595;187519872073.199;187510496351.554;187501121116.609;187491746357.544;187482372063.428;187472998235.172;187463624873.691;187454251979.919;187444879554.792;187435507599.260;187426136114.280;187416765100.824;187407394547.916;187404993323.005;187395623318.803;187386253774.963;187376884704.287;187367516107.769;187358147986.419;187355747388.123;187346379869.301;187337012826.195;187327646259.831;187318280171.242;187308914561.474;187292584463.104;187276255781.560;187259928505.825;187250565761.121;187241203459.265;187231841600.940;187222480186.833;187213119229.566;187203758741.894;187194398736.702;187192000185.009;187182640821.495;187173281953.428;187163923582.041;187154565708.580;187145208334.296;187142810577.401;187133453818.914;187124097524.750;187114741683.885;187105386297.107;187096031365.217;187086676889.018;187077322869.328;187067969306.971;187058616202.778;187049263557.591;187039911372.259;187030559647.644;187021208384.611;187011857572.112;187002507210.899;186993157301.730;186990761331.567;186988365400.075;186979016220.832;186969667528.204;186960319323.334;186950971607.370;186941624381.478;186932277634.911;186922931356.812;186913585536.212;186897289918.564;186880995716.927;186864702932.244;186855360007.632;186846017552.435;186836675567.615;186827334054.143;186817993012.995;186808652445.167;186799312339.745;186789972685.702;186780633471.901;186771294698.995;186761956367.649;186752618490.430;186743281080.038;186733944149.308;186731551546.234;186729159009.510;186726766540.235;186717430554.772;186708095047.511;186698760007.598;186689425435.979;186680091333.595;186670757701.405;186661424540.384;186652091851.505;186642759635.760;186633427894.150;186624096627.683;186614765825.476;186605435476.543;186596105569.769;186586776105.832;186577447085.414;186568118509.208;186558790377.909;186549462692.226;186540135452.866;186530808672.437;186521482363.678;186512156527.561;186502831165.072;186493506277.204;186484181864.976;186474857929.412;186465534471.552;186456211492.440;186446888993.141;186437566974.723;186428245438.266;186418924384.865;186409603803.736;186400283683.981;186390964014.591;186381644796.330;186379256762.258;186376868740.697;186367550174.420;186358232061.861;186341984795.462;186332667933.475;186323351525.476;186314035572.263;186304720074.636;186295405033.404;186286090449.381;186276776323.392;186267462668.148;186258149484.626;186248836773.816;186239524536.709;186230212774.315;186220901487.653;186211590677.750;186202280345.646;186192970492.391;186183661119.049;186174352226.687;186165043816.394;186155735877.389;186146428398.781;186137121369.568;186127814778.633;186118508626.612;186109202914.146;186099897641.891;186090592810.502;186081288432.503;186071984520.543;186062681087.412;186053378146.023;186044075697.577;186027855806.362;186011637338.370;186002337003.628;185993037135.449;185983737734.769;185974438802.528;185965140339.681;185955842347.191;185946544826.032;185944162321.187;185941779826.165;185939397341.458;185937014879.405;185927718248.742;185918422096.318;185909126423.189;185899831230.419;185890536507.234;185881242242.739;185871948425.931;185862655057.546;185853362138.329;185844069669.031;185834777650.411;185825486083.237;185816194980.126;185806904353.827;185797614205.380;185788324535.838;185779035334.415;185769746590.210;185760458292.211;185744263063.523;185728069244.551;185718783023.370;185709497280.412;185700212016.740;185690927233.417;185681642931.524;185672359100.308;185663075728.900;185653792806.328;185644510333.336;185635228310.684;185625946739.134;185616665631.292;185607384999.885;185598104857.787;185588825206.162;185579546046.200;185577168286.167;185574790570.122;185572412887.068;185570035225.895;185560757017.590;185551479249.534;185542201922.384;185532925036.786;185523648593.359;185514372592.741;185505097035.566;185495821934.302;185486547301.546;185470375956.922;185454206043.574;185444933554.830;185435661543.684;185426389999.312;185417118910.771;185407848267.015;185398578068.701;185389308316.498;185380039011.073;185370770164.925;185361501790.681;185352233889.280;185342966461.676;185333699508.834;185324433031.723;185315167019.509;185305901461.240;185296636345.852;185287371685.796;185278107493.647;185268843770.307;185252691419.123;185236540485.968;185227278870.920;185218017733.413;185208757062.639;185199496847.673;185190237077.478;185180977764.515;185171718921.375;185162460548.985;185153202648.273;185150830273.760;185148457934.887;185146085632.350;185143713366.855;185134456436.808;185125199962.251;185115943932.152;185106688347.161;185097433219.742;185088178550.686;185078924340.798;185069670590.887;185060417313.563;185051164509.782;185041912180.501;185032660314.890;185016528585.859;185000398245.202;184991148411.728;184981899026.193;184972650089.297;184963401601.738;184954153576.008;184944906012.951;184935658913.411;184926412278.245;184917166108.320;184907920416.292;184898675214.956;184889430505.450;184880186288.927;184870942554.760;184861699292.205;184852456490.414;184843214138.422;184833972225.152;184824730751.201;184815489717.171;184806249125.050;184797008972.127;184794641080.761;184785401501.957;184776162399.393;184766923774.056;184757685626.949;184748447959.081;184739210771.469;184729974065.149;184720737841.161;184711502088.774;184702266797.148;184693031955.321;184683797552.234;184674563576.698;184665330040.963;184656096957.410;184646864338.547;184637632197.005;184628400533.792;184619169349.916;184609938646.398;184600708424.271;184591478684.579;184582249428.377;184573020657.754;184563792360.794;184554564526.044;184545337143.419;184536110201.882;184526883702.057;184517657644.569;184508432030.055;184499206859.155;184489982132.520;184480757850.801;184471534014.661;184462310636.529;184453087728.963;184443865304.644;184434643364.630;184425421909.983;184423058921.606;184413838060.168;184404617649.231;184395397689.516;184386178181.746;184376959128.545;184360884594.003;184351666791.788;184342449446.998;184333232560.429;184324016132.880;184314800165.158;184305584658.083;184296369612.481;184287155029.187;184277940909.046;184268727252.909;184259514061.638;184250301336.092;184241089077.145;184231877285.671;184222665962.562;184220305560.382;184217945184.137;184208734543.720;184199524346.464;184190314593.009;184181105284.008;184171896420.117;184162688001.999;184153480030.326;184137424882.029;184121371123.686;184112165187.607;184102959705.731;184093754678.811;184084550107.618;184075345992.926;184066142335.518;184056939136.183;184047736407.457;184038534161.999;184029332412.608;184020131172.210;184010930442.125;184001730211.961;183992530471.212;183990173101.487;183987815766.942;183985458456.497;183976259525.049;183967061021.309;183957862945.756;183941824949.903;183925788323.992;183916592227.369;183907396565.912;183898201340.177;183889006550.730;183879812198.143;183870618282.992;183861424805.861;183852231779.060;183843039215.028;183833847126.334;183824655525.674;183815464414.162;183806273792.922;183797083663.089;183787894025.815;183778704882.251;183769516221.853;183760328033.952;183751140307.775;183741953032.434;183732766196.933;183723579790.156;183714393812.587;183698376967.539;183682361504.186;183673177531.285;183663994006.551;183654810930.681;183645628304.378;183643275122.917;183640921959.020;183638568813.156;183629386953.077;183620205538.667;183611024570.592;183601844061.236;183592664023.108;183583484468.846;183574305399.522;183565126816.214;183555948720.015;183546771112.024;183537593993.357;183528417365.137;183519241216.795;183510065537.646;183494066618.287;183478069085.035;183468895449.690;183459722268.187;183450549541.273;183441377258.010;183432205419.040;183429855082.798;183427504760.947;183418333569.312;183409162819.645;183399992512.576;183390822648.744;183381653228.792;183372484253.366;183363315734.817;183354147685.621;183344980106.692;183335812998.957;183326646363.352;183317480200.819;183308314524.005;183299149345.682;183289984655.373;183274004927.768;183258026604.493;183255678582.980;183253330580.723;183244168178.586;183235006198.602;183225844640.974;183216683505.899;183207522805.260;183198362551.069;183189202755.461;183180043419.008;183170884542.304;183161726125.937;183152568170.507;183143410676.619;183134253644.884;183125097075.919;183115940970.348;183106785328.800;183097630151.916;183088475452.010;183072513218.438;183056552411.439;183054206959.324;183051861562.393;183042709293.565;183033557483.448;183024406120.945;183015255194.843;183006104705.488;182996954653.233;182987805038.436;182978655861.450;182969507122.644;182960358822.381;182951210961.031;182942063538.970;182932916568.240;182923770061.009;182914624029.568;182905478486.343;182896333432.221;182893989999.415;182891646628.362;182882502348.842;182873358534.441;182864215174.360;182848272574.019;182832331341.751;182823189966.036;182814049005.288;182804908459.855;182795768341.740;182786628663.079;182777489436.123;182768350673.259;182759212375.348;182750074543.259;182740937177.868;182731800280.065;182722663850.745;182713527890.819;182704392401.193;182695257382.801;182686122836.575;182683782076.949;182681441361.770;182665514653.073;182649589337.710;182640457121.261;182631325370.633;182622194086.695;182613063258.682;182603932875.713;182594802926.802;182585673412.491;182576544333.332;182567415689.885;182558287482.709;182549159712.372;182540032379.450;182530905484.523;182528566569.390;182526227676.620;182517101448.130;182507975690.337;182498850404.168;182489725590.563;182480601250.468;182471477384.841;182462353994.647;182453231080.867;182444108644.486;182428202623.660;182412298015.690;182403177703.210;182394057865.469;182384938491.797;182375819571.418;182366701093.437;182357583046.850;182348465420.539;182339348214.905;182330231430.349;182321115067.283;182311999126.116;182309663021.199;182307326933.589;182298211645.575;182289096811.812;182279982433.050;182270868510.056;182261755043.598;182245864752.327;182236752542.602;182227640794.236;182218529519.680;182202643056.879;182193533097.418;182184423628.998;182175314652.757;182166206158.223;182163872116.015;182154764223.506;182145656801.512;182136549839.333;182127443326.161;182118337251.071;182116003803.756;182106898230.558;182097793071.147;182088688337.465;182079584041.579;182070480184.070;182061376765.528;182052273786.543;182036401755.046;182027299997.908;182018198696.216;182002329636.619;181993229612.317;181984130060.090;181981798190.785;181979466368.501;181970367554.117;181961269210.687;181952171339.137;181943073952.005;181933977050.355;181924880635.252;181915784696.180;181906689222.509;181897594203.502;181888499628.307;181879405485.963;181870311765.390;181861218467.006;181852125591.220;181843033138.462;181833941109.152;181818088141.285;181802236551.052;181793146548.200;181784057022.593;181774967975.221;181765879407.092;181756791319.217;181747703712.620;181745375007.091;181743046357.116;181733959517.443;181724873146.719;181715787234.251;181706701769.238;181697616740.762;181688532137.797;181679447960.783;181670364210.172;181661280898.006;181652198036.456;181643115626.233;181640788197.185;181631706351.622;181622624957.114;181606790499.273;181590957423.146;181581878068.799;181572799173.356;181563720737.620;181561394339.375;181552316497.041;181543239126.098;181534162227.469;181525085790.501;181516009804.444;181506934258.429;181497859141.473;181488784454.065;181479710196.693;181470636369.861;181461562985.635;181452490056.220;181450165033.747;181447840053.641;181432020829.777;181416202998.551;181407132403.771;181398062280.537;181388992629.779;181379923440.862;181370854703.036;181361786405.446;181352718548.683;181343651133.353;181334584160.059;181325517629.422;181316451542.072;181307385898.641;181298320699.770;181289255946.106;181280191638.305;181271127777.027;181262064362.939;181253001408.277;181243938925.396;181234876926.795;181225815413.531;181216754386.677;181214432510.532;181212110673.157;181196312109.494;181180514912.023;181171456147.844;181162397805.430;181153339885.179;181144282399.052;181135225359.147;181126168766.119;181117112620.642;181108056923.392;181099001675.058;181096681171.515;181094360693.648;181085306115.486;181076251984.122;181067198311.772;181058145110.785;181049092393.628;181033307971.212;181024256563.934;181015205634.583;181006155172.593;180997105167.305;180988055607.932;180972276563.424;180963228202.021;180954180267.507;180945132760.336;180936085692.512;180927039064.625;180917992877.270;180908947131.050;180899901826.573;180890856964.455;180881812545.318;180872768569.785;180863725038.495;180854681952.087;180845639322.743;180843322036.306;180834280016.971;180831962942.944;180822921572.456;180813880693.744;180804840296.406;180795800369.931;180780038132.122;180770999453.960;180761961216.908;180752923410.038;180743886022.306;180734849054.090;180725812505.773;180716776377.734;180707740681.892;180698705430.285;180689670635.074;180680636308.552;180671602451.612;180662569065.157;180653536150.100;180644503707.369;180635471737.890;180619723421.673;180603976480.901;180594946546.161;180585917046.568;180576887982.663;180567859354.984;180565545719.215;180563232106.905;180560918518.478;180551890666.314;180542863257.816;180533836293.613;180524809774.344;180515783712.169;180506758119.370;180497732996.847;180488708345.512;180479684166.283;180470660448.569;180461637181.667;180452614354.768;180443591968.455;180434570023.316;180425548519.952;180416527458.971;180407506840.992;180398486666.634;180382758939.463;180367032580.521;180351307590.609;180342290217.829;180333273299.283;180324256835.706;180315240839.350;180312930474.161;180310620165.015;180301604921.172;180292590131.933;180283575786.533;180274561874.099;180265548395.149;180256535350.201;180247522739.789;180238510575.939;180229498870.810;180227189599.989;180218178500.260;180209167870.936;180200157712.925;180191148015.642;180182138768.393;180173129960.375;180164121580.666;180155113618.238;180146106073.441;180137098958.119;180134790810.831;180132482694.737;180123476263.044;180114470293.076;180105464785.648;180096459741.587;180087455161.725;180078451046.903;180069447397.975;180060444204.304;180051441455.143;180035744049.376;180020047989.412;180004353264.346;179988659874.643;179979660592.188;179970661755.179;179961663375.739;179952665454.644;179943667992.674;179934670990.614;179925674449.266;179916678369.429;179907682751.921;179898687597.562;179889692907.180;179880698681.617;179871704921.710;179869400284.729;179867095691.239;179864791130.370;179855798189.276;179846805676.015;179837813591.046;179828821934.838;179819830707.857;179810839910.577;179801849554.947;179792859653.037;179783870217.050;179774881247.839;179765892746.272;179756904713.223;179747917149.579;179738930044.758;179729943388.065;179720957168.704;179711971375.755;179702986009.656;179694001082.318;179685016605.774;179676032580.722;179667049007.867;179658065887.926;179649083221.614;179640101009.661;179637799297.392;179628817645.794;179619836429.119;179604176601.915;179588518123.819;179579538888.839;179570560114.856;179561581802.667;179552603953.091;179543626566.942;179534649645.057;179525673188.268;179516697197.424;179507721661.922;179498746571.042;179489771913.960;179480797679.737;179471823868.776;179462850481.487;179453877529.729;179444905025.487;179435932980.870;179426961408.110;179417990319.574;179409019704.847;179400049553.417;179391079854.653;179382110609.272;179373141817.999;179364173481.561;179361875386.343;179359577324.757;179357279285.841;179348311726.765;179339344584.235;179330377858.576;179321411550.137;179312445670.696;179303480232.158;179294515246.556;179285550726.037;179276586671.441;179260956793.021;179245328278.345;179229701128.222;179214075343.459;179205114883.820;179196154881.202;179187195336.368;179178236238.642;179169277577.238;179160319352.701;179151361565.579;179142404227.853;179133447340.196;179124490914.714;179115534952.214;179106579453.518;179104284625.194;179101989840.424;179099695088.357;179090740405.315;179081786159.740;179072832352.184;179063878983.206;179054926053.373;179045973563.250;179037021513.407;179028069904.414;179019118736.849;179010168022.726;179001217774.184;178992267992.067;178983318677.234;178974369830.545;178965421441.446;178956473499.273;178947526004.675;178938578958.310;178929632360.844;178920686201.521;178911740480.899;178902795199.543;178893850346.601;178884905911.102;178875961893.392;178867018293.814;178864726422.030;178855783358.440;178846840749.953;178831248294.362;178815657204.761;178806716623.687;178797776513.033;178788836873.690;178779897706.567;178770959012.575;178762020792.639;178753083047.691;178744145767.258;178735208940.759;178726272557.504;178717336618.101;178708401123.167;178699466073.322;178690531469.202;178681597311.436;178672663589.257;178663730291.787;178654797408.032;178645864938.288;178636932882.865;178628001242.063;178619070027.593;178610139251.284;178607850670.613;178605562131.376;178603273634.097;178594343686.135;178585414195.447;178576485162.781;178567556588.895;178558628474.559;178549700820.555;178540773627.668;178531846896.693;178516282040.897;178500718563.515;178485156454.137;178476232554.732;178467309094.089;178458386072.772;178449463491.347;178440541350.382;178431619650.459;178422698392.162;178413777564.686;178404857157.110;178395937158.405;178387017568.817;178378098399.979;178369179663.642;178360261371.679;178351343536.093;178349058290.351;178346773093.634;178344487946.534;178335570980.867;178326654468.494;178317738410.126;178308822806.482;178299907658.283;178284363003.217;178268819717.448;178259906610.177;178250993966.174;178242081786.253;178233170071.240;178224258810.576;178215347993.604;178206437609.545;178197527647.515;178188618107.896;178179708991.075;178170800297.441;178161892027.389;178152984181.319;178144076748.253;178135169728.463;178119639295.895;178110733471.336;178108451271.463;178106169103.803;178103886968.761;178088359281.205;178079455071.113;178070551333.622;178061648069.641;178052745280.092;178043842954.528;178034941082.397;178026039653.029;178017138667.028;178008238124.996;177999338027.539;177990438375.278;177981539168.835;177972640408.837;177963742084.550;177954844185.129;177945946699.613;177930432815.622;177914920244.482;177906024648.193;177897129482.939;177888234760.546;177879340492.959;177870446692.255;177861553370.624;177852660540.393;177850381781.246;177848103085.258;177845824441.807;177836932513.283;177828041028.344;177819149976.235;177810259357.446;177801369172.475;177792479421.821;177783590105.991;177774701225.497;177765812780.856;177756924772.591;177748037201.227;177739150067.300;177730263359.989;177714768288.742;177699274543.141;177683782123.650;177674898086.492;177666014484.858;177657131330.602;177648248635.698;177639366412.253;177630484661.158;177621603383.318;177612722579.640;177610446907.472;177608171293.014;177605895725.591;177597015788.680;177588136288.964;177579257215.627;177570378557.739;177561500315.613;177552622489.553;177543745079.875;177534868086.786;177525991510.388;177517115362.111;177508239653.505;177499364396.241;177490489602.105;177481615283.014;177472741439.659;177463868072.746;177454995171.655;177446122725.663;177437250723.933;177428379155.517;177419508020.689;177410637319.728;177401767052.915;177386300653.738;177370835598.534;177355371887.765;177353099450.672;177350827038.399;177348554651.047;177339687453.569;177330820692.825;177321954369.122;177313088482.759;177304223034.033;177295358023.243;177286493450.697;177277629316.698;177268765621.555;177259902376.915;177251039594.539;177242177275.004;177233315418.892;177224454026.791;177215593099.301;177206732637.015;177197872640.546;177189013110.507;177173565380.992;177158119002.891;177149261473.423;177140404371.698;177131547686.605;177129278180.775;177127008674.667;177124739168.106;177115883148.206;177107027538.182;177098172338.060;177089317559.170;177080463212.958;177071609310.988;177062755864.951;177053902875.326;177045050342.612;177036198267.302;177027346649.898;177011912848.072;176996480410.502;176987630835.325;176978781724.988;176969933080.093;176961084901.249;176952237189.066;176943389944.170;176934543155.871;176925696813.371;176916850905.758;176908005433.324;176899160396.355;176890315795.146;176881471629.994;176879205365.411;176876939109.256;176874672850.130;176865829362.975;176856986271.511;176848143586.898;176839301320.419;176823883832.253;176808467689.790;176799627411.230;176790787590.597;176781948228.391;176773109325.126;176764270881.310;176755432897.489;176746595374.210;176737758312.029;176728921711.500;176720085573.196;176711249897.685;176702414685.544;176693579937.356;176684745642.413;176675911789.888;176667078368.850;176658245368.246;176649412786.254;176640580623.729;176638317389.768;176629485741.464;176620654511.889;176611823701.158;176602993309.361;176594163336.594;176585333782.946;176576504659.807;176567675978.683;176558847751.201;176550019989.103;176541192704.263;176532365897.405;176523539557.970;176514713675.293;176505888238.596;176497063236.986;176488238670.750;176479414540.176;176470590845.556;176461767587.182;176446383153.057;176431000059.055;176422178775.448;176413357934.857;176404537526.577;176402277267.386;176400017014.309;176397756767.400;176388937048.893;176380117757.281;176371298904.238;176362480501.553;176353662561.142;176344845083.784;176336028070.277;176327211521.420;176318395438.017;176303023528.799;176287652965.422;176278838877.883;176270025218.081;176261211975.155;176252399149.420;176243586741.185;176234774750.767;176225963178.483;176217152035.900;176208341334.709;176199531086.717;176190721303.856;176181911998.185;176179654619.249;176177397293.141;176175140009.155;176166331556.523;176157523532.818;176148715938.466;176139908773.910;176131102039.592;176122295735.956;176113489863.452;176098135759.766;176082782989.351;176073979063.677;176065175564.359;176056372491.765;176047569846.276;176038767639.536;176029965883.322;176021164589.518;176012363758.911;176003563392.287;175994763490.450;175985964054.202;175977165073.098;175968366536.577;175959568433.973;175944227814.903;175935430877.193;175933176637.046;175930922397.376;175928668158.176;175926413919.451;175911076031.671;175902280608.976;175893485629.631;175884691105.491;175875897048.533;175867103459.618;175858310339.622;175849517678.213;175840725464.966;175831933689.359;175823142340.764;175814351419.710;175805560926.743;175796770862.419;175781444341.870;175766119149.688;175757331035.562;175748543357.249;175739756115.435;175730969310.824;175722182944.138;175713397004.861;175704611493.596;175695826410.961;175687041757.587;175678257534.126;175669473741.238;175660690390.852;175651907495.031;175643125065.974;175634343104.798;175625561612.657;175616780590.713;175608000040.155;175599219962.192;175590440346.815;175581661183.912;175572882463.285;175564104174.631;175555326307.554;175546548849.809;175537771800.906;175535522659.890;175533273519.995;175531024381.645;175522247997.721;175513472032.997;175504696499.464;175495921409.245;175487146774.594;175478372596.670;175469598876.648;175460825615.725;175445528410.298;175436756366.123;175427984753.201;175419213572.531;175410442825.131;175401672512.037;175392902634.304;175384133193.014;175368842647.189;175360074410.528;175351306604.800;175342539219.847;175333772256.600;175325005716.006;175316239599.039;175307473917.916;175298708684.994;175289943912.768;175281179602.679;175272415756.202;175263652374.831;175248372387.731;175233093742.019;175224332356.099;175215571400.664;175206810876.910;175198050786.055;175189291129.338;175187046447.420;175184801776.416;175182557117.167;175180312470.527;175171553642.648;175162795257.886;175154037328.837;175145279857.005;175136522832.750;175127766257.554;175119010132.932;175110254460.423;175101499230.364;175092744433.007;175083990058.510;175075236096.941;175066482549.488;175057729417.361;175048976701.794;175033715335.733;175018455307.352;175009704574.544;175000954302.390;174992204492.597;174983455135.733;174974706222.275;174965957742.614;174957209687.052;174954967966.156;174952726259.969;174950484569.666;174941737234.871;174926485233.309;174911234562.599;174902489196.169;174893744285.864;174884999833.465;174876255840.779;174867512298.489;174858769197.177;174850026527.347;174841284290.614;174832542488.624;174823801123.046;174815060195.574;174806319707.934;174797579661.876;174788840047.978;174780100856.720;174771362089.654;174762623748.363;174753885834.447;174745148349.538;174742909329.531;174740670352.775;174738431420.971;174729694759.676;174720958552.150;174712222800.400;174703487495.262;174688256395.804;174673026619.062;174657798155.795;174649065532.246;174640333315.354;174631601506.701;174622870107.891;174614139131.707;174605408591.071;174596678499.047;174587948868.846;174579219702.634;174570491002.603;174561762759.841;174553034965.345;174544307610.037;174535580684.745;174526854191.411;174518128131.996;174515892088.750;174513656072.151;174511420083.957;174502694781.746;174493969894.087;174485245422.773;174476521369.617;174467797736.468;174459074536.388;174450351782.582;174441629477.267;174426421128.493;174411214126.205;174396008461.716;174387288926.526;174378569824.649;174369851147.036;174361132884.562;174352415039.139;174343697612.712;174334980618.437;174326264069.615;174317547968.564;174315314440.755;174313080955.200;174304365556.479;174295650610.409;174286936108.255;174278222041.197;174269508400.333;174260795187.803;174252082405.778;174243370056.464;174234658142.092;174232425694.330;174223714308.088;174215003350.089;174206292822.612;174197582727.965;174182395647.310;174167209896.078;174164978300.949;174156270294.568;174147562735.350;174138855614.663;174130148923.797;174121442653.961;174112736796.280;174104031352.965;174095326337.376;174086621762.639;174077917643.171;174069213981.290;174060510768.558;174051807996.463;174043105656.411;174034403739.730;174025702237.669;174023472422.192;174021242629.217;174012541764.251;173997370801.276;173982201172.056;173973502294.893;173964803861.280;173956105862.753;173947408301.874;173938711181.239;173930014492.377;173921318226.743;173912622375.701;173903926941.649;173895231927.066;173886537345.577;173877843199.850;173869149492.589;173860456226.540;173851763415.587;173843071073.765;173834379193.010;173825687765.183;173816996782.070;173814769744.378;173812542725.106;173810315726.597;173801625488.495;173792935679.392;173784246290.883;173775557314.482;173760407015.057;173745258011.736;173730110307.248;173721423954.550;173712738045.063;173704052581.740;173695367567.583;173686683005.632;173677998910.056;173669315272.944;173660632086.314;173651949353.196;173643267076.668;173634585248.749;173625903861.399;173617222906.488;173614998454.580;173612774009.737;173610549574.388;173601869317.434;173593189477.653;173584510046.738;173575831027.443;173567152422.556;173558474234.914;173549796467.379;173541119122.866;173532442215.402;173517313072.171;173502185283.077;173487058840.584;173478384739.722;173469711088.173;173461037878.082;173452365101.521;173443692761.556;173435020861.307;173426349403.924;173417678381.535;173409007786.185;173406785947.066;173404564110.349;173402342278.682;173393672362.743;173385002859.611;173376333783.357;173367665148.223;173358996968.595;173350329259.022;173341662012.019;173332995220.023;173324328875.406;173315662970.471;173306997497.448;173298332448.495;173289667815.702;173281003602.150;173272339810.958;173257233428.462;173242128351.120;173227024571.193;173218363415.059;173209702670.799;173207483268.141;173205263886.834;173203044540.982;173194384562.911;173185725039.722;173177065975.067;173168407372.646;173159749225.089;173151091524.958;173142434264.744;173133777447.929;173125121078.038;173116465158.651;173107809693.384;173099154674.849;173090500095.587;173081845948.074;173073192224.718;173064538917.851;173055886019.738;173047233522.578;173038581418.489;173029929699.520;173014844377.890;173006193796.239;172997543642.638;172995326971.387;172986677389.351;172978028257.508;172969379579.681;172960731359.739;172952083590.492;172943436275.738;172934789419.311;172926143014.063;172917497052.769;172908851528.144;172900206443.879;172891561803.711;172882917600.387;172874273826.583;172865630474.910;172856987537.908;172848345008.055;172839702888.795;172837488350.154;172835273822.276;172833059297.383;172824417855.679;172815776824.244;172807136206.681;172798496006.623;172783430831.142;172768366976.640;172753304458.240;172744667009.973;172736030038.958;172727393538.453;172718757501.655;172710121921.708;172701486791.683;172692852104.590;172684217853.383;172675584030.946;172666950630.096;172658317643.589;172649685075.147;172641052928.533;172632421207.565;172623789916.106;172621578194.296;172619366491.671;172617154800.764;172614943114.046;172606312611.237;172597682507.478;172589052806.474;172580423511.985;172571794627.815;172563166157.823;172548121471.038;172533078112.156;172518036096.625;172509410399.588;172500785177.816;172492160424.815;172483536134.030;172474912298.846;172466288912.588;172457665968.526;172449043459.861;172440421379.741;172431799721.245;172423178477.393;172414557652.157;172405937249.559;172397317273.680;172388697728.646;172380078618.629;172371459936.849;172362841676.460;172354223830.555;172345606392.164;172343398159.818;172341189935.455;172338981722.864;172336773525.883;172328156881.616;172319540660.408;172304517291.339;172289495247.592;172274474533.781;172265861067.295;172257248063.983;172248635528.499;172240023454.549;172231411835.793;172222800665.821;172214189938.170;172205579646.322;172196969783.699;172188360343.659;172179751319.505;172171142704.478;172162534491.758;172153926674.465;172145319245.650;172143113555.649;172140907856.518;172138702163.075;172130095387.809;172121489047.985;172112883159.199;172104277726.213;172095672753.850;172087068235.928;172072065224.756;172057063536.656;172042063165.467;172033461376.070;172024860016.424;172016259080.081;172007658571.540;171999058495.342;171990458856.095;171981859658.464;171973260907.164;171971057544.746;171968854211.370;171966650900.364;171958052898.532;171949455304.341;171940858111.215;171932261312.522;171923664912.621;171915068915.921;171906473326.893;171897878161.048;171889283434.071;171880689161.818;171872095349.271;171863502001.467;171854909112.527;171846316676.509;171837724687.431;171822743467.977;171807763563.432;171792784967.833;171784195677.798;171775606821.970;171773406003.979;171771205224.962;171769004478.559;171760416403.633;171751828747.460;171743241503.800;171734654666.352;171726068239.743;171717482228.656;171708896637.835;171700311461.095;171691726692.204;171683142324.868;171674558363.780;171665974813.686;171657391679.392;171648808976.740;171640226721.744;171638027494.682;171635828333.012;171633629230.813;171625047859.168;171610085169.675;171595123798.451;171580163739.841;171571585085.334;171563006859.621;171554429056.771;171545851681.769;171537274739.664;171528698235.567;171520122174.641;171511546551.159;171502971359.327;171494396593.313;171485822247.229;171477248315.131;171468674791.027;171460101679.835;171451528986.537;171442956716.174;171434384862.879;171432188330.350;171429991798.489;171427795271.928;171425598755.361;171417027666.051;171408457003.841;171399886784.964;171391317014.804;171382747698.811;171367806078.334;171352865778.974;171337926795.382;171329360210.821;171320794059.639;171312228336.007;171303663034.051;171295098147.843;171286533682.357;171277969642.634;171269406033.771;171260842860.933;171252280129.343;171243717833.332;171235155967.180;171226594525.125;171218033501.344;171209472889.969;171200912685.072;171192352880.685;171190159381.041;171187965887.031;171185772414.326;171183578978.778;171175020010.968;171166461509.383;171151538758.864;171136617340.340;171121697248.616;171113141527.052;171104586251.823;171096031417.477;171087477018.519;171078923049.402;171070369504.541;171061816389.236;171053263708.854;171044711468.835;171036159674.675;171027608331.941;171019057435.321;171010506979.459;171001956958.951;170999766198.793;170997575459.950;170995384736.500;170986835434.776;170978286544.595;170969738071.244;170961190020.073;170952642396.504;170944095195.071;170935548410.272;170927002036.551;170918456068.302;170903555011.014;170888655223.350;170873756699.760;170865213274.390;170856670254.720;170848127646.088;170845938561.745;170843749508.027;170841560490.262;170833018647.061;170824477247.031;170815936295.987;170807395788.834;170798855720.414;170790316085.518;170781776878.881;170773238095.186;170764699739.990;170756161818.911;170747624337.619;170739087301.849;170730550717.392;170722014590.099;170713478914.954;170704943686.892;170696408900.787;170687874552.143;170679340634.931;170670807143.753;170662274073.174;170653741418.128;170645209173.159;170636677332.762;170628145902.294;170619614887.172;170611084292.866;170602554124.903;170594024388.870;170585495090.406;170576966224.294;170568437785.258;170559909767.963;170551382167.031;170542854977.013;170540669950.111;170538484921.640;170536299885.600;170527773329.527;170519247171.247;170510721416.039;170502196069.244;170493671147.215;170485146666.480;170476622632.822;170461759993.944;170446898676.082;170432038674.321;170423517386.318;170414996537.646;170406476123.104;170397956137.436;170389436575.327;170380917442.322;170372398744.026;170363880486.092;170361697774.421;170359515112.758;170357332506.557;170355149950.411;170346632658.409;170338115799.548;170329599368.566;170321083360.133;170312567768.873;170304052589.347;170295537826.972;170287023487.218;170278509575.617;170269996097.749;170261483059.262;170246639397.068;170231797035.598;170216955969.739;170208445599.516;170199935649.080;170191426113.024;170182916996.786;170174408305.864;170165900045.804;170157392222.219;170148884829.876;170140377863.485;170138198108.742;170136018367.023;170133838643.373;170131658942.874;170123152802.473;170114647079.142;170106141767.448;170097636861.893;170089132367.828;170080628290.649;170072124624.919;170063621365.143;170055118505.764;170046616052.068;170038114009.393;170029612383.136;170014788743.187;169999966390.674;169985145320.358;169970325526.942;169961827242.883;169959649525.469;169957471827.542;169955294165.081;169946796632.257;169938299540.109;169929802894.274;169921306700.425;169912810964.289;169904315680.751;169895820844.638;169887326450.709;169878832493.661;169870338968.127;169861845868.671;169853353189.794;169844860936.816;169836369115.099;169834193215.488;169832017358.155;169829841548.201;169821350545.071;169812859982.877;169804369856.207;169795880159.566;169787390887.402;169778902034.095;169770413593.951;169761925572.095;169753437973.697;169744950803.965;169736464068.153;169727977771.563;169719491919.543;169711006506.603;169702521527.192;169694036975.684;169691863037.884;169689689116.554;169687515205.457;169679031358.847;169664238529.062;169649446987.325;169634656738.886;169619867789.020;169611387345.473;169602907339.415;169594427765.159;169585948616.956;169577469888.982;169568991575.346;169560513670.086;169552036178.037;169543559104.080;169535082453.130;169526606230.149;169518130440.138;169509655088.139;169501180168.375;169492705674.995;169484231602.073;169475757943.622;169467284693.569;169458811856.650;169456640885.142;169454469935.540;169452299012.264;169450128119.777;169441656126.802;169433184545.015;169424713368.241;169416242590.230;169407772204.660;169399302215.988;169390832628.720;169382363447.395;169373894676.582;169365426320.902;169356958384.999;169348490873.571;169340023780.479;169325260303.486;169310498101.322;169302032864.102;169293568022.325;169291398872.506;169282934500.818;169274470537.651;169266006998.379;169257543887.637;169249081210.097;169240618970.473;169232157173.513;169223695824.014;169215234926.801;169206774475.898;169198314465.249;169189854888.733;169181395740.143;169179228144.085;169177060565.252;169174893007.660;169166434582.618;169157976579.338;169149519002.285;169141061855.960;169132605144.904;169124148873.693;169115693036.093;169100949171.469;169086206593.894;169071465297.122;169063012086.987;169054559295.944;169046106928.325;169037654988.510;169029203480.910;169020752409.971;169018586937.887;169016421502.121;169014256095.855;169005805794.827;168997355902.267;168988906411.473;168980457326.505;168972008651.459;168963560390.457;168955112547.664;168946665127.272;168938218133.511;168929771570.638;168921325442.954;168919161204.578;168916996991.179;168902270429.248;168887545136.398;168872821105.974;168864377756.893;168855934797.985;168847492233.043;168839050065.890;168830608300.379;168828445026.925;168826281766.452;168817840605.525;168809399879.196;168800959591.655;168792519747.126;168784080349.860;168775641404.154;168767202903.503;168758764841.326;168750327210.965;168741890005.680;168739728006.471;168737566022.004;168729129416.429;168720693203.582;168712257387.095;168703821970.632;168689113820.015;168674406940.952;168659701337.414;168651268512.812;168649107531.860;168646946584.513;168638514414.788;168630082677.992;168621651378.080;168613220519.044;168604790104.903;168596360128.890;168587930584.152;168579501463.766;168571072760.717;168562644478.751;168560484744.625;168558325032.111;168549897369.921;168541470117.469;168533043278.355;168524616856.203;168516190854.674;168501499114.980;168486808665.901;168478384581.833;168476225927.203;168474067320.880;168465643935.404;168457220997.087;168448798509.817;168440376466.706;168431954860.786;168423533684.994;168408850215.399;168400430193.771;168392010582.891;168383591375.391;168381434038.412;168379276715.636;168370858105.591;168356179150.103;168347761683.536;168339344638.478;168330928018.425;168322511826.889;168314096067.414;168311939622.971;168309783222.744;168301368148.311;168286695341.690;168278281459.104;168269868006.298;168261454975.951;168253042360.666;168244630152.954;168236218345.233;168227806929.848;168213140460.830;168198475238.820;168196320198.021;168194165159.586;168192010126.150;168183600757.401;168175191789.121;168166783224.377;168158375066.260;168149967317.878;168141559982.368;168133153062.885;168124746562.609;168116340484.747;168107934832.524;168099529620.004;168091124861.380;168082720560.196;168074316709.213;168065913301.105;168063759978.979;168061606688.523;168059453421.697;168057300170.378;168042648492.892;168027998074.842;168013348919.325;168004948495.446;167996548485.747;167988148893.233;167979749720.918;167971350971.844;167962952649.077;167954554755.697;167946157294.817;167937760269.565;167929363683.094;167920967538.580;167912571839.228;167904176588.260;167895781788.924;167887387444.491;167878993547.479;167870600090.311;167868449451.276;167866298839.528;167864148246.761;167855755517.944;167847363186.536;167832729950.307;167824338706.718;167815947855.957;167807557400.576;167799167343.147;167790777686.257;167782388432.511;167773999584.531;167765611144.961;167757223116.455;167748835501.693;167740448303.368;167732061524.194;167723675166.902;167715289234.247;167706903728.992;167698518653.934;167690134011.877;167681749805.653;167673366038.109;167664982701.340;167662834529.669;167654451714.335;167646069303.333;167637687288.205;167635539389.528;167627157839.862;167618776680.049;167610395923.150;167602015571.573;167593635627.745;167585256094.108;167576876973.113;167568498267.237;167560119978.969;167551742110.809;167543364665.286;167534987644.934;167526611052.309;167518234889.984;167509859160.550;167501483866.612;167493109010.793;167484734595.743;167476360624.117;167461759602.029;167447159886.298;167438787890.688;167436642671.046;167434497496.824;167432352359.403;167430207250.059;167421836144.998;167413465437.478;167405095118.849;167396725191.130;167388355656.347;167379986516.535;167371617773.745;167363249430.040;167354881487.492;167346513948.186;167338146814.217;167329780087.701;167321413770.755;167313047865.518;167304682374.135;167296317298.765;167287952641.588;167273367915.060;167258784468.014;167244202302.903;167242059379.731;167239916507.628;167237773688.552;167235630913.726;167233488174.266;167225126763.506;167216765756.745;167208405145.176;167200044930.643;167191685115.001;167183325700.104;167174966687.830;167166608080.054;167158249878.676;167149892085.591;167141534702.714;167133177731.968;167124821175.284;167110250611.100;167095681320.444;167081113305.527;167072759381.014;167064405898.096;167056052858.958;167053912364.895;167051771924.132;167049631538.406;167047491198.736;167039139114.373;167030787445.443;167022436183.055;167014085318.210;167005734841.815;166997384755.401;166989035060.509;166980685758.688;166972336851.493;166963988351.217;166955640259.538;166947292578.147;166938945308.738;166936806196.055;166934667105.393;166926320455.791;166911767179.956;166897215173.539;166882664438.469;166874320399.623;166865976788.732;166857633607.590;166849290858.001;166840948541.779;166832606671.471;166824265249.022;166822127755.984;166819990322.436;166811649617.536;166803309342.287;166794969487.687;166786630044.644;166778291003.949;166769952356.295;166767815562.729;166765678773.464;166757340685.915;166749002992.121;166740665693.359;166732328790.914;166723992286.070;166715656180.116;166707320474.342;166698985180.761;166690650311.505;166682315868.100;166680180052.767;166678044274.146;166669710491.178;166655179636.442;166640650063.946;166626121775.470;166617790654.470;166609459973.900;166601129735.376;166592799940.515;166584470590.948;166576141688.308;166567813234.240;166559485219.684;166551157635.481;166542830472.361;166540696735.109;166538563007.522;166536429290.232;166528102806.456;166519776721.444;166511451036.251;166503125751.935;166494800869.555;166486476390.174;166478152314.850;166463638266.160;166449125476.315;166440803246.735;166438670649.138;166430348933.239;166422027630.175;166413706741.047;166405386266.958;166397066209.006;166388746568.301;166380427345.950;166378295518.038;166376163719.883;166367845133.841;166359526966.076;166351209217.657;166342891889.649;166334574983.120;166326258488.455;166317942395.922;166303442284.660;166295127310.594;166286812742.158;166278498582.678;166270184831.897;166261871490.691;166253558559.942;166245246040.523;166236933933.314;166234803878.969;166226492287.489;166218181108.597;166209870343.143;166207740624.596;166199430377.310;166191120543.766;166182811124.794;166180681748.189;166172372849.618;166164064365.840;166155756297.661;166147448645.877;166139141411.285;166130834595.107;166116350734.578;166101868142.845;166093563209.416;166091435009.465;166083130617.193;166074826650.034;166066523108.804;166058219994.312;166056092300.688;166053964646.077;166045662192.476;166037360164.342;166029058551.758;166020757344.697;166012456533.016;166004156117.128;165995856097.439;165981384040.427;165966913240.756;165952443699.204;165944146250.945;165942019970.198;165939893715.301;165937767486.324;165929470766.917;165921174459.090;165912878563.327;165904583080.110;165896288009.919;165887993353.223;165879699110.494;165871405282.197;165863111868.791;165854818870.733;165846526288.468;165844401242.331;165842276222.202;165840151228.020;165838026259.717;165829734507.143;165821443163.849;165806986271.082;165792530638.062;165778076265.385;165763623153.644;165755335116.947;165747047500.483;165738760304.629;165730473529.763;165722187176.257;165713901244.472;165705615734.766;165697330647.500;165689045983.014;165680761741.657;165672477923.761;165664194529.649;165662071901.393;165659949305.586;165657826731.365;165655704178.402;165653581646.351;165645299184.323;165637017125.650;165628735470.373;165620454218.532;165612173370.156;165603892925.261;165595612883.861;165587333245.955;165572896756.914;165558461520.719;165544027537.638;165529594807.937;165521318440.920;165513042486.307;165504766944.127;165496491814.401;165488217097.143;165479942792.353;165477822388.184;165475702004.803;165473581641.660;165471461298.184;165469340973.796;165461067571.224;165452794569.428;165444521968.219;165436249767.389;165427977966.721;165419706565.985;165411435564.945;165403164963.345;165388744354.311;165374324991.498;165359906885.546;165351638846.851;165343371245.152;165335104080.530;165326837353.059;165318571062.804;165310305209.820;165308186972.584;165306068781.720;165303950636.742;165301832537.130;165299714482.369;165291449683.229;165283185310.084;165274921362.803;165266657841.231;165252249416.774;165237842263.831;165223436382.512;165215175497.230;165206915043.543;165198655021.320;165190395430.423;165182136260.087;165173877499.442;165165619137.493;165157361173.766;165149103607.777;165140846439.033;165138730547.919;165136614662.710;165134498782.374;165132382905.865;165124126465.293;165115870409.831;165107614738.862;165093220036.294;165078826562.538;165064434317.249;165056181112.124;165047928305.657;165039675907.976;165031423929.309;165023172369.387;165021057731.769;165018943123.641;165016828544.211;165008577722.670;165000327311.717;164992077310.997;164983827720.148;164975578538.808;164967329766.607;164959081403.174;164944699555.154;164930318959.625;164915939616.496;164913826333.147;164911713075.075;164909599841.432;164901354349.278;164893109262.988;164884864582.154;164876620306.364;164868376435.206;164860132968.262;164851889905.105;164843647245.310;164835404988.444;164827163134.073;164825050896.465;164822938672.422;164820826460.965;164812585279.330;164804344501.753;164789976001.369;164775608769.989;164761242807.703;164746878114.614;164738640695.763;164730403709.597;164722167155.985;164713931034.801;164711820351.679;164709709713.035;164701474269.188;164693239252.762;164685004663.577;164676770501.449;164674660389.191;164672550317.395;164664316816.295;164656083737.016;164647851079.317;164639618842.954;164631387027.709;164623155633.370;164614924659.734;164606694106.589;164598463973.725;164596354972.968;164594246004.573;164586016513.596;164571667418.357;164557319580.146;164542972998.997;164534746096.297;164526519614.181;164518293541.853;164510067868.411;164501842582.840;164493617684.590;164485393183.666;164483285595.281;164481178036.737;164479070507.217;164476963005.900;164468739336.013;164460516073.752;164452293218.776;164444070770.730;164435848729.262;164427627094.019;164413291733.145;164398957619.958;164384624754.409;164376405670.777;164368186998.270;164359968736.576;164351750885.389;164349644981.382;164347539099.127;164345433237.720;164343327396.244;164335110348.186;164326893699.641;164318677450.181;164310461599.369;164302246146.765;164294031091.924;164279707271.919;164265384689.941;164251063345.842;164242850817.700;164234638712.975;164232534222.066;164230429777.454;164228325378.482;164226221024.489;164218009844.275;164209799088.445;164201588756.822;164193378849.236;164185169365.512;164176960305.471;164168751668.938;164160543455.730;164152335665.665;164144128298.560;164129817766.959;164115508498.780;164107303018.395;164099097960.591;164096995470.460;164094893022.739;164086688629.276;164078484646.860;164070281064.627;164062077882.159;164053875099.024;164045672714.792;164037470729.025;164029269141.284;164021067951.123;164012867158.091;164004666761.733;164002565392.645;164000464032.657;163998362680.707;163990162938.312;163981963594.036;163973764657.973;163965566140.305;163957368040.807;163949170359.235;163940973095.356;163932776247.208;163924579818.474;163916383808.737;163902093094.016;163893898221.868;163885703768.137;163877509732.590;163869316114.994;163861122915.113;163852930132.703;163844737767.525;163836545808.796;163828354245.624;163826255152.142;163824156067.196;163822056989.712;163813866076.912;163805675550.537;163797485410.014;163783204956.197;163768925725.930;163754647729.475;163746460085.065;163744361880.877;163742263702.408;163734076674.580;163725890052.372;163717703835.415;163709518023.342;163701332615.775;163699234962.567;163697137328.611;163688952516.925;163680768103.755;163672584088.661;163658314415.754;163644045977.779;163629778785.121;163627681994.713;163625585233.039;163617403964.105;163609223103.390;163601042650.554;163592862615.759;163584682998.771;163582586812.614;163580490659.760;163572311679.821;163564133112.264;163555954956.793;163547777213.111;163539599880.917;163531422959.904;163523246449.764;163515070349.566;163506894660.519;163498719382.126;163490544512.334;163482370062.322;163474196031.872;163466022420.765;163457849228.776;163449676455.682;163441504111.753;163433332196.867;163425160710.906;163416989653.748;163414895802.059;163406725293.512;163398555201.147;163396461607.683;163388292010.423;163380122804.523;163371953989.469;163363785564.743;163355617529.823;163347449884.173;163339282627.263;163325042096.750;163310802789.143;163302637319.620;163294472262.162;163286307616.472;163284215300.696;163282123010.573;163280030745.190;163277938503.629;163269774675.699;163261611248.701;163253448222.226;163239215125.424;163224983263.488;163216822049.681;163208661239.138;163200500831.474;163192340826.294;163184181223.204;163182090142.600;163179999078.631;163177908030.283;163163681446.538;163149456101.004;163141298609.821;163133141536.709;163124984881.431;163116828643.756;163108672823.438;163106582731.611;163104492672.918;163102402646.517;163094247570.173;163086092903.261;163071874397.504;163057657134.868;163049504307.296;163041351892.278;163033199889.525;163025048298.738;163016897119.627;163014808251.879;163012719421.700;163010630638.760;162996418791.984;162982208203.726;162974059260.927;162965910747.163;162957762662.321;162949615006.275;162941467778.907;162933320969.619;162931233358.565;162929145760.403;162927058174.074;162918912038.415;162904708260.732;162890505704.588;162882361345.087;162874217376.850;162866073799.373;162857930612.134;162849787825.064;162847701132.512;162845614466.463;162837472295.978;162823275528.803;162815134473.452;162806993824.374;162798853581.214;162790713743.617;162782574311.220;162774435283.658;162766296660.556;162764211021.725;162762125403.485;162760039804.842;162751901874.725;162737712520.580;162723524394.833;162709337507.816;162701202110.732;162693067135.532;162684932582.026;162676798450.019;162674713992.691;162672629573.615;162670545191.990;162662411823.427;162654278868.691;162646146327.500;162638014199.579;162629882484.649;162621751182.423;162613620292.614;162605489814.934;162591313373.158;162583184024.667;162575055077.566;162566926520.817;162564843535.809;162556715490.890;162548587866.024;162540460661.007;162532333875.618;162524207509.649;162516081562.886;162507956035.107;162499830926.094;162491706235.622;162483581963.470;162475458109.409;162467334673.210;162465253119.042;162463171598.540;162461090110.839;162459008655.074;162450886074.376;162442763890.728;162434642103.680;162420480691.358;162406320508.197;162392161554.128;162378003829.066;162363847332.936;162355729458.652;162353649302.056;162351569167.108;162349489052.822;162341371875.767;162333255095.094;162325138710.351;162317022721.076;162308907126.808;162300791916.641;162298712361.385;162296632818.224;162288518178.111;162280403946.881;162272290134.635;162264176741.143;162256063766.189;162247951209.552;162245872376.247;162243793579.217;162241714817.618;162227570163.946;162213426753.279;162199284585.695;162185143650.857;162171003948.736;162162895703.748;162154787868.146;162146680441.616;162138573423.836;162136496053.822;162134418709.892;162132341391.097;162124235087.527;162116129184.394;162108023681.290;162099918577.796;162091813873.496;162083709567.964;162075605660.775;162073529048.149;162071452451.605;162069375870.086;162061272624.307;162053169747.065;162045067248.010;162036965136.898;162022838187.259;162008712477.026;161994588006.266;161980464775.035;161966342783.406;161964267476.746;161962192205.438;161960116968.622;161958041765.430;161949944067.745;161941846777.449;161933749894.205;161925653417.676;161917557347.515;161909461683.378;161901366424.910;161893271571.759;161885177123.571;161877083079.980;161868989440.623;161866915418.106;161864841417.000;161862767436.296;161860693474.967;161852600625.380;161838489790.428;161824380178.655;161810271789.971;161796164624.287;161782058681.512;161773969720.194;161765881160.122;161757793000.898;161755720321.355;161753647671.185;161751575059.834;161743487647.245;161735400668.968;161727314124.950;161719228015.136;161711142329.077;161703057056.215;161694972185.882;161692900460.230;161690828755.844;161688757071.707;161680672892.590;161672589107.394;161664505715.647;161650411329.862;161636318165.761;161622226223.253;161608135502.253;161600055294.484;161597984700.078;161595914124.153;161593843565.656;161585764038.432;161577684902.266;161569606156.655;161561527801.089;161553449835.052;161545372258.028;161537295069.492;161529218268.913;161527148433.930;161519072114.049;161510996200.721;161502920703.794;161494845630.408;161486770981.956;161478696758.289;161470622959.258;161462549584.708;161454476634.487;161446404108.441;161438332006.411;161430260328.238;161422189073.766;161408115866.511;161394043908.601;161391976083.091;161389908303.561;161387840569.242;161385772868.989;161383705191.533;161375636353.925;161361567255.096;161347499364.520;161339432283.375;161331365588.759;161323299280.127;161315233356.917;161307167818.579;161299102664.544;161291037894.240;161282973507.094;161274909502.520;161266845879.934;161258782637.932;161250719774.714;161242657299.814;161240591149.294;161238525032.268;161230463187.257;161222401757.998;161214340744.252;161206280145.776;161198219962.320;161190160193.633;161182100839.468;161174041899.568;161165983373.678;161157925261.536;161149867562.879;161141810277.785;161133753405.220;161125696944.737;161123632471.304;161115576534.293;161107521006.950;161099465888.969;161091411180.049;161083356879.873;161075302988.132;161067249494.155;161059196387.169;161045154777.119;161031114366.458;161029051046.360;161026987724.656;161024924410.390;161022861112.708;161020797840.848;161012746926.213;161004696418.131;160990659464.323;160976623739.817;160968575054.541;160960526778.833;160952478912.397;160944431454.924;160936384406.102;160928337765.622;160920291533.168;160912245708.420;160898216877.160;160884189276.134;160876145277.382;160874083980.783;160872022713.213;160869961473.707;160867900261.293;160865839074.991;160857796002.042;160855734943.786;160847692370.991;160839650193.027;160825627694.704;160817586606.233;160809545913.393;160801505615.736;160787486415.899;160779447208.379;160771408396.830;160763369980.791;160755331959.794;160747294343.697;160733279850.940;160725243344.619;160717207244.315;160709171549.691;160695160397.291;160687125813.960;160679091637.458;160677032831.840;160674974050.165;160666940477.941;160658907304.339;160656848747.956;160654790215.769;160652731706.756;160650673219.889;160642640842.051;160634608854.093;160626577255.514;160618546045.799;160610515224.438;160602484790.905;160594454744.682;160580453488.010;160566453439.930;160552454600.312;160544427016.949;160536399845.473;160528373095.906;160520346778.371;160512320892.811;160504295439.154;160496270417.345;160488245827.312;160480221668.996;160478165513.622;160476109408.306;160468085924.823;160460062847.165;160452040164.516;160444017876.259;160441962211.635;160439906569.062;160431884869.257;160423863558.534;160415842636.383;160407822102.289;160399801955.731;160391782196.185;160383762823.119;160375743835.997;160367725234.281;160359707017.418;160351689184.859;160343671736.046;160335654670.411;160327637987.390;160319621686.403;160311605777.149;160297629189.202;160283653835.307;160269679725.901;160255706861.249;160247694242.997;160245640973.951;160243587754.480;160241534583.822;160239481461.205;160237428385.852;160235375356.974;160233322373.776;160225311033.280;160217300085.457;160209289519.542;160201279334.959;160193269531.117;160185260107.422;160177251063.279;160169242398.078;160161234111.213;160153226202.060;160145218670.000;160131256623.159;160117295774.027;160103336122.402;160089377678.350;160075420452.039;160067416752.241;160065365689.327;160063314661.358;160061263667.412;160059212706.559;160057161777.856;160055110880.354;160053060013.087;160045057482.387;160037055349.138;160029053612.919;160021052273.313;160013051329.895;160005050782.235;159997050629.899;159989050882.704;159981051550.573;159967103848.061;159953157380.925;159939212149.362;159925268153.583;159911325393.803;159903330056.403;159895335122.802;159887340582.381;159885291958.944;159883243347.952;159881194748.239;159879146158.632;159877097588.191;159875049046.073;159873000531.279;159865007049.866;159857013962.932;159849021270.037;159841028970.727;159833037064.551;159825045551.049;159817054429.761;159803121009.367;159789188799.964;159775257801.517;159761328013.994;159753340056.728;159745352507.828;159737365377.241;159729378664.794;159727332019.693;159725285413.718;159723238845.978;159721192315.582;159719145821.621;159717099363.184;159715052929.089;159701128488.302;159687205235.337;159673283180.204;159665299668.414;159657316541.602;159649333799.275;159641351451.159;159633369496.850;159625387935.945;159617406768.032;159609425992.695;159601445609.516;159593465618.076;159579551643.160;159565638876.251;159551727317.346;159549682845.548;159547638401.077;159545593982.923;159543549590.055;159541505231.671;159539460906.826;159537416614.574;159535372353.956;159527395682.183;159519419407.414;159505511907.342;159491605620.180;159477700545.983;159469726758.078;159461753371.830;159453780397.131;159445807843.986;159437835712.280;159429864001.898;159421892712.724;159413921834.403;159405951356.474;159397981268.364;159384084461.684;159370188859.881;159362220537.681;159360178603.062;159358136680.878;159356094769.952;159348127099.213;159340159821.298;159338118126.044;159336076472.548;159334034859.944;159331993287.357;159324026877.125;159316060871.343;159308095269.678;159300130071.799;159292165277.363;159278277712.785;159264391366.452;159250506238.500;159242543955.713;159234582081.169;159226620614.580;159218659555.652;159210698893.864;159202738618.590;159194778729.313;159186819225.511;159178860106.659;159170901372.224;159157024359.794;159149066692.512;159141109404.655;159133152505.673;159131113487.555;159129074508.783;159127035578.615;159124996696.213;159122957860.729;159120919071.304;159112963289.135;159105007906.957;159097052924.395;159089098341.072;159081144156.609;159073190370.615;159065236972.491;159051369227.480;159037502684.825;159023637344.513;159009773206.538;159001822944.529;158993873076.069;158985923600.722;158977974518.047;158970025827.592;158962077528.909;158954129621.541;158952092928.372;158950056257.508;158948019618.015;158945983008.843;158938035921.472;158930089247.566;158922142986.735;158914197138.588;158906251692.136;158898306645.442;158890361998.137;158882417749.844;158874473900.183;158866530448.775;158858587395.226;158850644739.148;158842702480.147;158834760617.823;158826819151.770;158818878081.586;158805031834.239;158791186795.898;158777342966.645;158769404373.932;158767370129.829;158765335899.997;158763301673.036;158761267447.614;158759233222.394;158757199006.180;158749261307.305;158741324002.077;158733387100.235;158725450601.446;158717514505.378;158703677006.658;158689840723.068;158676005654.767;158668072064.232;158660138881.210;158652206105.415;158644273736.557;158636341774.344;158634309224.183;158632276704.449;158624345360.736;158616414408.249;158608483836.194;158606451615.947;158598521508.643;158590591777.758;158582662422.678;158574733442.789;158566804847.629;158558876646.829;158550948850.129;158543021467.361;158535094498.311;158521272921.029;158507452569.022;158493633442.572;158491602722.517;158489572046.400;158487541413.318;158485510822.363;158483480262.440;158481449722.336;158473525950.349;158459709830.112;158445894904.148;158437972878.220;158430051239.006;158422129986.007;158414209118.720;158406288636.637;158392478308.019;158378669173.666;158370750439.492;158362832092.747;158354914132.944;158352885097.929;158350856083.486;158348827098.613;158340909838.696;158332992998.714;158330964294.477;158328935620.247;158321019397.442;158313103567.930;158305188131.282;158291386614.485;158277586299.064;158269672633.426;158261759363.208;158253846488.007;158245934007.417;158238021921.023;158230110228.408;158222198929.156;158208404636.473;158194611546.018;158180819657.857;158178792885.821;158176766135.654;158174739406.193;158172712706.421;158170686035.279;158168659391.702;158166632774.610;158164606182.911;158156698149.734;158148790504.042;158135002588.971;158121215870.646;158107430349.090;158093646024.321;158085741506.110;158077837381.105;158069933648.881;158062030309.010;158054127371.183;158046224845.190;158038322740.927;158030421048.121;158022519756.400;158014618855.278;158012594231.386;158010569621.800;158002669288.953;157994769341.005;157986869777.295;157978970597.163;157976946363.203;157969047658.459;157961149333.417;157953251387.509;157945353820.170;157937456630.730;157929559818.409;157921663382.419;157913767321.968;157905871646.368;157892104740.305;157878339046.866;157864574576.171;157862551723.229;157860528915.587;157858506152.152;157856483431.813;157854460753.459;157852438115.956;157844545660.747;157830784266.442;157817024083.176;157803265110.958;157795375160.682;157787485618.545;157779596484.089;157771707756.849;157763819426.233;157755931481.535;157748043911.947;157740156706.536;157738135552.624;157730248802.335;157722362441.289;157714476478.782;157712455641.571;157710434834.706;157702549480.898;157694664519.886;157686779951.045;157678895773.765;157671011987.403;157663128591.328;157649382987.893;157635638580.694;157621895369.600;157614014420.294;157606133864.469;157604114397.649;157602094950.436;157600075521.468;157598056109.358;157596036712.710;157588157007.438;157574417881.495;157560679960.035;157546943253.099;157539066049.554;157531189279.050;157523312941.392;157515437036.385;157507561553.723;157505543512.984;157503525501.811;157501507508.834;157493632727.397;157485758332.088;157472028315.787;157464154977.953;157456282023.122;157448409450.577;157440537259.601;157432665449.461;157424794019.421;157416922968.740;157409052296.666;157395328851.645;157381606589.447;157373737638.542;157371721120.177;157369704609.798;157367688105.925;157365671607.035;157363655121.687;157355786975.209;157347919223.595;157340051876.327;157332184942.994;157318468131.706;157304752539.489;157291038166.471;157277325012.787;157269461334.455;157261598076.321;157253735238.060;157245872819.349;157243858119.832;157241843456.739;157239828818.791;157237814194.582;157235799572.585;157233784951.246;157225923475.554;157218062361.277;157204355558.436;157190649921.235;157176945449.266;157163242152.172;157155384051.437;157147526345.518;157139669043.890;157131812156.132;157123955681.856;157121942387.561;157119929134.305;157117915920.943;157115902746.313;157108047142.788;157094349986.581;157080654035.400;157072800229.047;157064946827.875;157057093831.412;157049241239.174;157041389050.682;157033537265.441;157025685882.700;157017834900.839;157015823141.633;157013811406.161;157005961009.106;156998110977.448;156990261300.220;156982411976.422;156968725719.688;156955040617.666;156941356679.899;156933509678.101;156931498746.271;156929487838.705;156927476964.116;156925466131.317;156917619951.465;156909774175.742;156901928803.662;156894083834.741;156880405266.217;156866727903.386;156853051746.281;156851041965.804;156849032222.831;156847022516.165;156839180374.898;156831338632.897;156823497289.627;156815656344.555;156801984712.277;156794144855.457;156786305395.826;156784296545.661;156776457589.754;156768619028.755;156760780862.082;156752943089.155;156745105699.326;156737268681.844;156735260412.426;156733252134.553;156725415620.611;156711751753.633;156698089071.860;156684427585.156;156676593510.896;156668759845.117;156666752427.499;156664745049.198;156656912017.318;156649079387.881;156641247160.392;156639240151.369;156631408432.904;156629401563.861;156621570348.116;156607915660.039;156594262169.268;156586432728.697;156578603684.657;156570775035.762;156562946783.364;156555118926.901;156547291465.807;156539464399.514;156537458758.033;156535453141.752;156533447539.297;156525621130.281;156511974775.501;156498329566.218;156484685511.893;156476861402.184;156474856374.096;156472851364.849;156465027826.920;156457204690.152;156449381954.041;156447377300.707;156433737680.701;156420099258.433;156412278410.312;156410274264.667;156402453923.487;156394633979.240;156386814431.372;156378995279.316;156371176522.510;156369172933.395;156367169370.417;156359351206.894;156345719315.627;156332088611.396;156324272200.799;156316456181.171;156308640551.885;156300825312.322;156298822618.892;156296819945.868;156294817301.864;156287002750.383;156273377162.562;156259752764.761;156246129556.868;156232507538.799;156230505732.139;156228503953.670;156226502202.080;156218691079.614;156210880344.597;156203069996.371;156195260034.274;156187450457.638;156179641265.786;156171832458.036;156169831393.546;156167830343.486;156160022094.749;156152214226.628;156138600321.214;156124987589.203;156111376040.328;156103570584.695;156095765538.878;156087960912.445;156080156705.074;156078156868.825;156076157083.298;156074157347.414;156066353919.719;156058550882.880;156050748226.269;156042945939.145;156035144020.664;156033144747.290;156025343270.804;156017542161.577;156003940009.211;155990339016.496;155976739183.052;155968940410.396;155961142023.541;155953344031.787;155951345680.167;155949347365.314;155947349085.994;155945350840.964;155937553685.835;155929756924.164;155921960555.346;155914164578.778;155900571429.514;155886979470.371;155873388701.290;155859799122.208;155852006278.163;155850009300.018;155848012350.262;155846015427.566;155844018540.565;155836226526.053;155828434917.415;155820643714.157;155812852905.822;155805062481.829;155797272443.578;155789482789.814;155775900590.683;155762319568.648;155748739723.515;155746744100.739;155744748498.116;155736961432.181;155729174746.362;155721388439.924;155713602512.115;155705816962.189;155703821821.184;155701826700.046;155699831607.312;155692046740.225;155684262281.192;155676478229.767;155668694585.495;155660911347.916;155647340372.542;155639558243.913;155631776523.659;155623995201.224;155616214265.949;155602647218.783;155589081338.477;155581302105.005;155573523249.154;155565744770.157;155557966667.238;155550188939.615;155542411596.449;155534634646.990;155532641765.871;155530648929.697;155528656137.296;155526663387.495;155524670679.092;155516894713.483;155509119137.573;155501343940.749;155493569122.263;155485794681.344;155478020617.227;155470246929.135;155456692561.403;155443139367.871;155435367395.743;155427595800.353;155419824580.796;155412053746.096;155404283305.374;155390734661.878;155382965313.491;155375196370.204;155367427831.531;155359659696.982;155351891966.065;155344124638.276;155336357713.113;155328591190.062;155326601072.946;155324610990.137;155322620940.346;155320630922.270;155318640934.593;155310875337.471;155303110127.960;155295345295.439;155287580829.175;155279816728.274;155272052991.829;155258515931.808;155244980032.945;155231445294.911;155217911717.376;155210150983.873;155202390633.793;155194630676.226;155186871120.354;155184882705.674;155182894344.237;155175135443.209;155167376952.060;155159618860.404;155151861167.693;155144103873.375;155136346976.889;155128590477.669;155120834375.144;155113078668.736;155105323357.862;155097568441.931;155089813920.347;155087826908.196;155085839911.554;155083852928.919;155081865958.771;155079878989.636;155077892019.859;155064371897.101;155050852927.836;155037335111.670;155023818448.200;155010302946.936;154996788617.488;154983275469.559;154969763512.963;154962015205.947;154960029687.185;154958044207.848;154956058766.679;154954073362.410;154952087993.761;154944340651.539;154936593701.347;154928847142.558;154921100974.531;154913355196.625;154905609808.189;154897864808.563;154890120197.084;154882375973.078;154874632135.867;154872647798.423;154864904409.491;154857161375.621;154849418695.678;154841676368.508;154828176717.940;154814678225.395;154801180900.430;154793440936.304;154785701374.055;154777962223.052;154770223482.866;154768240394.896;154766257352.331;154764274353.972;154756536376.769;154748798801.777;154741061628.468;154733324856.311;154725588484.764;154723606094.280;154721623739.205;154719641418.230;154711905768.021;154704170509.268;154690683070.844;154677196807.972;154663711710.589;154650227768.534;154642495535.462;154634763654.962;154627032125.863;154625050864.806;154617319750.343;154609588994.849;154601858607.235;154594128596.499;154586398971.733;154578669742.120;154570940916.951;154563212505.599;154561232051.081;154559251644.824;154557271285.647;154555290972.355;154547563453.351;154539836337.089;154526363164.118;154512891186.494;154499420404.281;154485950817.542;154478226883.663;154470503344.663;154462780189.920;154455057410.765;154447334997.591;154439612949.443;154431891265.368;154424169944.392;154416448985.535;154414470472.496;154412491955.647;154410513433.211;154408534903.392;154400814570.650;154393094595.268;154385374986.080;154377655752.005;154369936902.062;154362218445.380;154348760536.456;154335303833.020;154327587188.530;154319870958.635;154312155142.812;154304439740.534;154296724751.268;154283272752.645;154275558875.767;154267845413.674;154260132356.076;154252419692.584;154244707412.699;154236995505.807;154229283971.070;154227307933.135;154225331901.593;154223355874.766;154221379850.958;154219403828.458;154211693043.195;154203982613.252;154196272537.606;154188562825.075;154180853484.580;154173144525.127;154159703133.190;154146262934.083;154132823937.761;154125117470.518;154117411415.815;154109705773.106;154102000541.845;154094295721.476;154080861406.849;154073157662.590;154065454301.941;154057751314.305;154050048698.836;154042346454.687;154034644581.001;154032670964.840;154030697354.178;154028723747.326;154026750142.563;154019048941.449;154011348096.937;154003647617.859;153995947513.140;153988247791.804;153980548462.955;153972849535.805;153970876598.176;153968903697.727;153961205390.579;153953507484.688;153945809979.397;153932388499.592;153918968201.578;153905549085.346;153892131150.892;153878714398.212;153865298817.453;153857605779.527;153849913114.108;153842220810.503;153840249609.452;153838278424.312;153836307263.332;153834336134.844;153832365037.426;153830393969.653;153822702633.023;153815011679.147;153807321107.299;153799630916.757;153791941106.792;153784251676.664;153776562625.633;153768873952.950;153761185657.858;153747780242.052;153740093022.131;153732406209.838;153724719804.561;153717033795.855;153709348173.166;153701662935.677;153688262787.199;153680578600.570;153672894799.320;153665211382.781;153657528350.295;153649845701.187;153642163434.778;153634481550.387;153632513053.885;153630544574.890;153628576111.819;153626607663.082;153624639236.875;153622670831.654;153614989873.444;153607309288.796;153593917303.587;153580526489.717;153567136847.174;153553748375.948;153540361076.033;153526974947.423;153513589990.113;153511623018.160;153509656075.023;153507689159.238;153505722269.324;153503755403.782;153501788561.098;153494113664.022;153486439142.982;153478764997.457;153471091236.735;153463417870.205;153455744897.528;153448072318.360;153440400132.440;153432728330.535;153425056911.999;153417385876.183;153409715222.421;153396340509.855;153388670902.476;153381001677.500;153373332834.498;153365664373.038;153357996292.684;153356031320.163;153354066366.082;153352101429.053;153350136507.672;153342469164.801;153334802191.885;153327135598.181;153313768025.682;153306102508.005;153298437390.899;153290772673.877;153277408330.410;153264045147.576;153256382151.076;153254418485.909;153252454842.934;153244792417.225;153237130368.829;153229468697.301;153227505348.480;153225542014.952;153223578695.283;153221615388.036;153213954433.591;153206293844.798;153198633630.896;153185277214.625;153171921978.948;153158567934.056;153150910188.077;153143252858.402;153135595944.729;153127939436.940;153125977499.500;153124015594.021;153116359680.659;153108704132.189;153101048948.066;153093394127.747;153085739670.682;153078085576.316;153070431844.092;153068470506.443;153066509167.779;153064547826.496;153062586480.974;153060625139.367;153058663809.904;153045319523.843;153031976409.120;153018634475.865;153005293724.542;152991954155.615;152978615769.567;152970966972.126;152963318574.020;152961358626.872;152959398725.768;152957438869.595;152955479057.238;152947831528.024;152940184385.649;152932537620.051;152924891221.069;152917245188.208;152909599520.972;152907640279.746;152905681061.602;152903721865.192;152901762689.147;152899803532.092;152897844392.630;152884514247.296;152871185257.769;152857857424.288;152844530747.113;152831205226.495;152817880862.700;152810240085.628;152808282050.554;152806324054.555;152804366106.206;152802408214.190;152800450377.499;152792810600.297;152785171221.666;152777532231.723;152769893620.482;152762255377.855;152760298066.649;152758340752.979;152750703009.271;152743065621.473;152735428588.656;152722112543.302;152708797638.523;152695483884.222;152687849171.589;152680214854.062;152678258452.231;152676302084.541;152674345749.752;152672389446.612;152670433173.863;152668476930.224;152666520714.411;152658887507.651;152651254679.955;152643622231.058;152630314211.839;152617007364.221;152603701698.363;152590397224.435;152582767924.328;152575139033.788;152573184158.869;152571229319.694;152569274505.090;152567319703.763;152565364904.300;152563410104.933;152555782123.205;152548154491.912;152540527210.327;152532900277.718;152525273703.096;152511975945.753;152504350401.841;152496725244.177;152489100471.963;152481476084.408;152479522214.729;152477568374.384;152475614561.895;152467990859.877;152460367536.877;152452744592.482;152445122026.296;152437499847.651;152435546562.981;152427924901.407;152420303645.041;152412682793.294;152405062345.572;152397442301.270;152389822651.110;152382203384.994;152374584492.841;152366965974.210;152359347828.649;152357395645.415;152355443474.116;152353491313.095;152351539160.678;152349587015.173;152347634874.871;152340017583.498;152332400654.410;152319119703.663;152305839887.509;152292561215.722;152279283698.182;152271669751.510;152264056205.854;152256443061.105;152254492079.828;152252541137.767;152250590233.560;152248639365.825;152246688533.167;152244737734.178;152242786967.434;152240836231.500;152238885524.918;152231273744.383;152223662370.341;152210391064.331;152197120936.622;152183851978.007;152170584179.197;152162975894.989;152155367978.724;152147760429.988;152140153248.350;152138203870.388;152136254496.882;152134305126.104;152132355756.311;152130406385.738;152128457022.330;152126507674.107;152118901415.984;152111295549.198;152103690073.037;152096084986.786;152088480289.722;152086531501.375;152078927299.202;152071323481.445;152063720047.893;152056116998.330;152048514332.546;152040912060.043;152033310190.425;152020055466.439;152012454688.763;152004854302.239;151997254296.433;151995306790.357;151993359300.595;151991411825.552;151989464363.625;151987516913.183;151985569472.589;151983622040.176;151981674614.275;151974075645.904;151960825991.549;151947577489.352;151934330149.399;151926733560.459;151919137371.356;151911541582.096;151903946192.675;151901999858.556;151900053563.564;151898107296.639;151890512606.710;151882918280.270;151875324306.622;151873378287.905;151871432285.504;151869486307.539;151867540362.218;151865594448.135;151858001334.498;151850408606.565;151837169802.082;151829578117.403;151821986815.147;151814395895.178;151806805357.354;151799215201.543;151791625427.605;151778391765.967;151770803039.604;151763214691.166;151761270144.465;151759325620.811;151757381128.395;151755436665.805;151753492231.604;151745904761.989;151738317674.600;151736373467.630;151734429287.349;151732485132.296;151724898719.802;151722954685.106;151715368746.481;151713424830.283;151705839362.589;151698254265.430;151685028740.443;151671804361.238;151664220944.589;151656637901.733;151649055232.498;151641472946.406;151639529946.276;151631948154.797;151624366750.548;151616785732.831;151614843080.232;151612900459.943;151610957870.574;151609015310.721;151607072778.965;151599492656.205;151591912914.999;151589970609.507;151588028330.458;151586086076.401;151578506998.669;151576564835.270;151574622665.098;151561407872.988;151548194204.827;151540616890.391;151533039947.805;151525463386.623;151517887216.499;151510311447.190;151497102279.771;151489527614.689;151481953357.113;151480012472.608;151478071642.268;151476130864.944;151474190139.485;151472249464.721;151470308839.471;151462735721.985;151455162988.790;151447590629.501;151445650321.195;151443710022.805;151441769732.726;151439829449.337;151437889171.001;151430317570.576;151422746328.011;151415175442.332;151407604910.717;151400034731.040;151392464903.029;151384895426.411;151377326300.913;151369757535.926;151362189140.948;151348992825.009;151341425475.686;151333858502.942;151326291906.008;151324352861.659;151322413840.049;151320474849.393;151318535897.990;151316596994.228;151314658136.916;151312719324.847;151305153884.992;151297588844.392;151290024202.531;151288085787.791;151286147418.358;151284209093.019;151282270810.554;151280332569.725;151272768902.990;151265205631.329;151252018056.241;151244455849.117;151236894034.122;151229332611.482;151221771581.412;151214210944.146;151206650690.243;151199090810.883;151191531292.114;151183972123.358;151176413293.936;151174476399.719;151172539482.182;151170602539.410;151168665569.455;151166728570.363;151164791540.146;151162854486.467;151160917417.061;151158980339.748;151151422307.164;151143864639.634;151136307345.972;151128750435.077;151126813747.230;151124877108.559;151117320838.760;151109764984.851;151102209557.010;151089035930.573;151075863514.274;151068309965.736;151066374442.529;151064439004.084;151056886200.567;151049333815.407;151047398718.310;151045463667.747;151043528652.891;151041593662.796;151039658696.071;151032107233.906;151024556147.712;151017005436.771;151015070777.479;151013136140.624;151011201524.791;151009266928.553;151007332350.465;151005397789.073;150997848000.151;150984683821.464;150971520778.508;150958358871.983;150950811391.553;150943264281.313;150935717541.391;150933783824.557;150931850119.266;150929916414.346;150927982698.509;150926048960.342;150924115188.322;150916569150.945;150909023422.610;150901478011.569;150893932926.167;150886388174.818;150884454585.550;150882520997.495;150880587418.677;150878653857.197;150876720321.246;150874786819.107;150872853359.139;150870919949.799;150868986599.632;150861443148.009;150853900129.329;150846357543.506;150833206265.179;150820056185.569;150812515450.613;150804975150.568;150797435286.300;150789895858.691;150787964099.138;150786032419.394;150784100809.035;150782169257.530;150780237754.237;150772699417.815;150765161463.985;150757623882.509;150750086663.043;150742549804.778;150740618780.085;150738687765.682;150736756760.065;150734825761.710;150732894769.085;150730963780.643;150729032794.810;150727101810.008;150725170824.637;150723239837.084;150715704094.710;150708168703.364;150700633662.135;150693098970.100;150685564626.333;150672427486.914;150664894117.802;150657361096.243;150649828422.267;150642296095.909;150640365857.917;150638435619.910;150636505389.917;150634575166.411;150632644947.860;150630714732.704;150628784519.379;150626854306.292;150619322924.581;150611791896.926;150604261232.089;150596730938.925;150589201026.373;150581671503.474;150579741867.041;150577812289.601;150575882779.808;150573953346.411;150572023988.626;150570094705.663;150568165496.726;150566236361.010;150564307297.701;150551180829.108;150538055552.562;150524931469.603;150511808581.774;150498686890.649;150496758992.319;150494831170.964;150492903425.819;150490975746.484;150489048122.455;150487120543.113;150485192997.728;150483265475.462;150481337965.358;150479410465.974;150471887291.693;150464364481.085;150456842033.407;150449319947.921;150441798223.871;150439871146.924;150437944080.663;150436017023.634;150434089974.368;150432162931.375;150430235893.161;150428308858.209;150420788040.343;150413267576.936;150405747467.168;150403820657.067;150401893848.658;150399967030.778;150398040192.149;150396113321.378;150388593820.566;150381074618.382;150373555713.429;150371628863.555;150369701977.459;150367775062.805;150360256593.353;150352738458.892;150345220669.092;150332112748.589;150319005962.409;150311489832.306;150303974085.375;150302047893.685;150300121754.580;150298195676.671;150296269659.048;150294343700.796;150292417800.985;150290491958.683;150288566182.553;150281051893.704;150273538043.280;150266024631.357;150258511658.019;150256586594.892;150254661621.209;150252736736.376;150250811939.786;150248887230.833;150246962608.903;150239451022.623;150231939856.422;150224429100.585;150222504950.356;150220580850.767;150218656800.813;150216732799.488;150214808845.768;150212884938.624;150205375304.795;150197866069.418;150190357232.171;150182848792.732;150180925409.320;150179002074.427;150177078787.041;150175155546.127;150173232350.655;150171309199.556;150163801853.132;150156294878.129;150143204805.782;150130115856.416;150122610508.370;150115105504.571;150107600844.146;150105678385.835;150103755920.997;150101833448.006;150099910965.220;150097988470.979;150096065963.602;150094143441.406;150092220902.670;150084717059.787;150077213551.156;150069710375.802;150062207532.748;150054705021.005;150052782724.931;150050860391.057;150048938007.840;150047015563.619;150045093046.615;150043170454.508;150041247794.547;150039325074.052;150037402300.425;150029900233.110;150022398482.469;150014897056.947;150007395965.071;149999895215.455;149997972671.343;149996050138.640;149994127625.511;149992205140.209;149990282691.066;149988360286.510;149986437935.059;149978938283.265;149971439036.305;149963940193.960;149956441756.009;149954519961.556;149952598232.082;149950676576.328;149948755003.116;149946833521.366;149944912140.096;149942990868.413;149935493970.690;149927997535.701;149920501563.847;149913006055.527;149905511011.152;149903590777.004;149901670660.219;149899750660.506;149897830767.993;149895910972.704;149888417277.341;149880924007.690;149873431155.386;149871511870.784;149864019572.890;149856527693.420;149849036232.326;149841545189.550;149834054565.047;149826564358.762;149824645982.378;149822727677.459;149820809443.228;149818891278.888;149816973183.634;149815055156.651;149813137197.120;149811219304.204;149809301477.061;149807383705.263;149799895283.123;149792407249.747;149784919595.214;149777432309.494;149769945392.033;149762458842.273;149760541685.445;149758624547.983;149756707428.530;149754790325.724;149752873238.177;149750956164.494;149749039103.261;149747122053.047;149739636559.990;149732151428.863;149724666659.043;149717182240.345;149709698162.461;149702214414.993;149700297774.832;149698381106.988;149696464409.599;149694547680.781;149692630918.630;149690714121.223;149688797286.608;149686880412.820;149684963497.868;149677480396.814;149669997604.780;149662515120.610;149655032933.569;149647551032.807;149640069407.368;149638152521.085;149636235550.262;149634318492.426;149632401345.069;149630484105.657;149628566771.631;149626649349.955;149619167873.740;149611686677.370;149604205768.943;149596725156.639;149589244848.728;149587327438.889;149585409993.120;149583492509.436;149581574985.825;149579657420.252;149577739820.220;149570260037.739;149562780589.000;149555301482.663;149547822727.492;149545905376.221;149543988045.511;149542070743.518;149540153478.484;149532675501.470;149525197930.181;149517720764.476;149515803928.818;149508327297.183;149500851069.217;149487816165.794;149480341057.083;149478424917.903;149476508874.318;149474592935.443;149467118791.931;149459645123.519;149452171940.544;149444699253.451;149442784283.556;149440869483.385;149438954843.734;149437040355.289;149435126008.646;149433211794.293;149425740932.541;149418270555.976;149410800665.507;149403331262.045;149395862346.512;149388393919.843;149386481099.981;149384568424.483;149382655893.409;149380743497.272;149378831226.480;149376919071.342;149375007031.611;149367540490.961];
%Lower Initial population
%n1 = n + 1;
%D0 = ones(n1,1)*10^8; %Initial population of cells

%BFU-E Cell Density
d_BFUE = BFU_E_start:h:BFU_E_end;
BFU_E = 10^8*exp(B_BFUE*d_BFUE);
BFU_E_D = trapz(d_BFUE, BFU_E);

%CFU-E Cell Density
d_CFUE = CFU_E_start:h:CFU_E_end;
n_CFUE = round(CFU_E_start/h)+1:round(CFU_E_end/h)+1;
CFU_E_D = [];
CFU_E_D = [CFU_E_D trapz(d_CFUE, D0(n_CFUE))];

%Progenitors Cell Density
Progenitors = [];
Progenitors = [Progenitors BFU_E_D+CFU_E_D];

%Erythroblasts Cell Density
d_Ery = Erythroblasts_start:h:Erythroblasts_end;
n_Ery = round(Erythroblasts_start/h)+1:round(Erythroblasts_end/h)+1;
Erythroblasts_D = [];
Erythroblasts_D = [Erythroblasts_D trapz(d_Ery, D0(n_Ery))];

%Reticulocyte Cell Density
d_Reti = Reticulocyte_start:h:Reticulocyte_end;
n_Reti = round(Reticulocyte_start/h)+1:round(Reticulocyte_end/h)+1;
Reticulocyte_D = [];
Reticulocyte_D = [Reticulocyte_D trapz(d_Reti, D0(n_Reti))];

%Erythrocytes  Cell Density
d_RBCs = Erythrocytes_start1:h:Erythrocytes_end3;
n_RBCs_1 = round(Erythrocytes_start1/h)+1:round(Erythrocytes_end1/h)+1;
n_RBCs_2 = round(Erythrocytes_start2/h)+1:round(Erythrocytes_end2/h)+1;
n_RBCs_3 = round(Erythrocytes_start3/h)+1:round(Erythrocytes_end3/h)+1;
n_RBCs_tot= round(Erythrocytes_start1/h)+1:round(Erythrocytes_end3/h)+1;
Erythrocytes_D = [];
Erythrocytes_D = [Erythrocytes_D trapz(d_RBCs, D0(n_RBCs_tot))];

%Set initial Values
for ii = 1:7*mesh
    D0(ii) = BFU_E(ii);
end
D0(n_CFUE(1)) = Cell_qin;
t = 0;
dt = Total_Span/n;
time = [0];
D0new = D0;
am_vect = [0.005];

%Initial EPO Value
Et = 9.8002;
Et_end = 9.8002;
EPO_tot = [Et]; %To monitor over time
EPO_end = [Et_end]; %To monitor over time

%For making the gif
t_settest = 1;
t_settest2 = 1;

while t < tset
    %Calculate CFU-E Apoptosis Rate
    alpha_CFUE = (a1-b1)/(1+exp(k1*Et-c1)) + b1;
    
    %Calculate Reticulocyte Maturation Velocity
    v_Reti = (a2-b2)/(1+exp(-k2*Et+c2)) + b2;
    v_Reti_set =  3 - (3-0.75)*(v_Reti-0.5)/(2-0.5);
    v_Reti_set_rd = round((Reticulocyte_start+v_Reti_set)/h)+1;
        
    %Calculate Time step growth for each cell type
    %CFU-E Day 7 to 13
    for ii = n_CFUE
    D0new(ii) = D0(ii)*exp((B_CFUE-alpha_CFUE)*dt);
    end
    
    %Erythroblasts Day 13 to 18
    for ii = n_Ery
        D0new(ii) = D0(ii)*exp((B_Ery)*dt);
    end
    
    %Reticulocyte Day 18 to 21
    for ii = n_Reti(1):v_Reti_set_rd
        D0new(ii) = D0(ii)*exp(-alpha_Reti*dt);
    end
    
    %Erythrocytes Day 21 to 35
    for ii = v_Reti_set_rd+1:n_RBCs_2-1
        D0new(ii) = D0(ii)*exp(-alpha_RBCs*dt);
    end
    
    %Erythrocytes Day 35 to 42, Neocytolysis possible
    if Et < tau_E
            am = alpha_RBCs + min(c_E/Et^k_E , b_E);
        else
            am = alpha_RBCs;
        end
        am_vect = [am_vect am];
    for ii = n_RBCs_2
        D0new(ii) = D0(ii)*exp(-am*dt);
    end
    
    %Erythrocytes Day 42 to 141
    for ii = n_RBCs_3
        D0new(ii) = D0(ii)*exp(-alpha_RBCs*dt);
    end
    
    %Age cell populations
    D0(n_CFUE(1)) = Cell_qin;
    
    %Day 0 to 18
    for ii = n_CFUE(1):n_Ery(end)
        D0(ii+1) = D0new(ii);
    end
    
    %Day 18 to 21 (maturation velocity)
    D0(n_Reti(1)+1:n_Reti(end)) = 0;
    for ii = n_Reti(1):v_Reti_set_rd
        D0(ii+1) = D0new(ii);
    end
    for ii = v_Reti_set_rd+1:n
        D0(ii+1) = D0new(ii);
    end
    
    %Calculation of cell density of each stage
    
    %Progenitors
    CFU_E_D1 = trapz(d_CFUE, D0(n_CFUE));
    Progenitors = [Progenitors BFU_E_D+CFU_E_D1];
    
    %Erythroblasts
    Erythroblasts_D = [Erythroblasts_D trapz(d_Ery, D0(n_Ery))];
    
    %Reticulocyte
    Reticulocyte_D = [Reticulocyte_D trapz(d_Reti(1):h:d(v_Reti_set_rd), D0(n_Reti(1):v_Reti_set_rd))];
    
    %Erythrocytes
    Erythrocytes_D = [Erythrocytes_D trapz(d(v_Reti_set_rd+1):h:d(n), D0(v_Reti_set_rd+1:n))];
    
    %Calculate New EPO concentration
    Erythrocytes_Mt = trapz(d_RBCs, D0(n_RBCs_tot));
    Mt = 1e-8*Erythrocytes_Mt/TBV; %Total RBC population
    Et_end_in = (b3-a3)/(1+exp(k3*Mt - c3)) + a3; %Rate of EPO Prod, /day\
    if t >= Time_Delay
        Et_ex_in = interp1(t_EPO,EPO_ex,t-Time_Delay);
    else
        Et_ex_in = 0;
    end
    Et_end = Et_end + (1/TBV*(Et_end_in) - c_end_deg*Et_end)*dt;
    Et = Et_end + Et_ex_in;
    EPO_tot = [EPO_tot Et];
    EPO_end = [EPO_end Et_end];
    
    %Add time step to time
    t = t + dt;
    time = [time t];

    %-----To make a gif of the dynamics of red blood cell production-----%
    if abs(t-t_settest) < 1/mesh/10
        %Overall Gif
        figure(2)
        set(gcf, 'Position', [900, 200, 900, 700])
        plot(d,D0, 'r*', [CFU_E_start CFU_E_start],y1, 'g--',...
            [Erythroblasts_start Erythroblasts_start], y1, 'b--', ...
            [Reticulocyte_start Reticulocyte_start], y1, 'k--',...
            [d(v_Reti_set_rd+1) d(v_Reti_set_rd+1)], y1, 'r--')
        title(['Time: ' num2str(t-Time_Delay, '%4.2f') ' days.   ', 'EPO: ' num2str(Et, '%4.2f') ' mU/mL.   ',...
            'Marrow Transit Time: ' num2str(d(v_Reti_set_rd)-18, 3) ' days'])
        xlabel('Cell Population Maturity (days)')
        ylabel('Total Cell Population')
        legend('Cell Population at given maturity','BFU-E end. CFU-E start.', ...
            'CFU-E end. Erythroblasts start.', 'Erythroblasts end. Reticulocyte start.', ...
            'Reticulocyte end. Red blood cell start.')
        set(gcf,'color','w'); % set figure background to white
        set(gca,'fontsize',12)
        drawnow;
        frame = getframe(2);
        im = frame2im(frame);
        [imind,cm] = rgb2ind(im,256);
        outfile = 'Dynamic Pop Mesh.gif';
        
        % On the first loop, create the file. In subsequent loops, append.
        if abs(t - t_settest2) < 1/mesh/10
            imwrite(imind,cm,outfile,'gif','DelayTime',0,'loopcount',inf);
        else
            imwrite(imind,cm,outfile,'gif','DelayTime',0,'writemode','append');
        end
        
        %Gif of Marrow Transit Time
        figure(3)
        set(gcf, 'Position', [0, 200, 900, 700])
        plot(d,D0, 'r*', [Reticulocyte_start Reticulocyte_start], y1, 'k--',...
            [d(v_Reti_set_rd+1) d(v_Reti_set_rd+1)], y1, 'r--')
        title(['Time: ' num2str(t-Time_Delay, '%4.2f') ' days.   ', ...
            'EPO: ' num2str(Et, '%4.2f') ' mU/mL.   ', 'Marrow Transit Time: ' num2str(d(v_Reti_set_rd)-18, '%4.2f') ' days'])
        xlabel('Cell Population Maturity (days)')
        ylabel('Total Cell Population')
        xlim([Reticulocyte_start-1 Reticulocyte_end])
        legend('Cell Population at given maturity', 'Erythroblasts end. Reticulocyte start.', ...
            'Reticulocyte end. Red blood cell start.')
        set(gcf,'color','w'); % set figure background to white
        set(gca,'fontsize',12)
        drawnow;
        frame = getframe(3);
        im = frame2im(frame);
        [imind,cm] = rgb2ind(im,256);
        outfile = 'Single Injection Mat Vel.gif';
        
        % On the first loop, create the file. In subsequent loops, append.
        if abs(t - t_settest2) < 1/mesh/10
            imwrite(imind,cm,outfile,'gif','DelayTime',0,'loopcount',inf);
        else
            imwrite(imind,cm,outfile,'gif','DelayTime',0,'writemode','append');
        end
        t_settest = t_settest + 1;
    end
end

%Figure of the EPO and red blood cell concentration cver time
figure(4)
fig = figure(4);
left_color = [0 0 0];
right_color = [1 0 0];
set(fig,'defaultAxesColorOrder',[left_color; right_color]);
xlabel('Time (days)')
yyaxis left
    plot(time-Time_Delay,EPO_tot, '-k', 'LineWidth',2); hold on;
    plot([-Time_Delay tset-Time_Delay], [3.3 3.3], '--k', 'LineWidth',2);
    plot([-Time_Delay tset-Time_Delay], [23.4 23.4], '--k', 'LineWidth',2);
    ylabel('EPO Concentration (mU/mL)')
    
    plot([0 0], [0 35],'--g' , 'LineWidth',2)
    plot([Detection_Window Detection_Window],...
        [0 35], '--b', 'LineWidth',2)

yyaxis right
    plot(time-Time_Delay,Erythrocytes_D/10^12, '-r', 'LineWidth',2);
    plot([-Time_Delay tset-Time_Delay], [40 40], '--r', 'LineWidth',2)
    ylabel('Red Blood Cell Population (# of cells x10^{12})')
    ylim([23 42])

legend('EPO total', 'Lower Natural EPO Limit','Upper Natural EPO Limit', 'Vector injection (Day 0)',... 
    ['Doping DNA detection limit (Day ' num2str(Detection_Window, '%4.0f') ')'], ...
    'Red blood cells', 'Stroke risk (RBC Population > 40x10^{12})')
set(gcf,'color','w'); 
set(gca,'fontsize',14);
set(gcf, 'Position', [500, 200, 800, 700])
xlim([-Time_Delay tset-Time_Delay])
toc

max_RBCs = max(Erythrocytes_D/10^12)
Pass_30 = interp1(Erythrocytes_D(1:10000)/10^12,time(1:10000),30)-Time_Delay
Pass_30_2 = interp1(Erythrocytes_D(10000:end)/10^12,time(10000:end),30)-Time_Delay
Max_day = interp1(Erythrocytes_D/10^12, time, max_RBCs) - Time_Delay